#include <sourcemod>
#include <tf2_stocks>
#include <dueldbnatives>
#include <morecolors>

#pragma newdecls required
#pragma semicolon 1

public Plugin myinfo =
{
    name = "Scoreboard Database Plugin",
    author = "SpookyMonster",
    description = "Plugin to handle scoreboard database",
    version = "1.0",
    url = ""
};

Database g_dbScoreboard = null;

bool g_bIsRanked[MAXPLAYERS + 1];
int g_iPoints[MAXPLAYERS + 1];
int g_iUnrankedKills[MAXPLAYERS + 1];
float g_fLastQuery[MAXPLAYERS + 1];

public void OnPluginStart()
{
    Database.Connect(Database_Connect, "default");
    HookEvent("player_death", Event_PlayerDeath, EventHookMode_Post);
    RegConsoleCmd("sm_rank", Command_Rank);
    RegConsoleCmd("sm_leaderboard", Command_Leaderboard);
    RegConsoleCmd("sm_top", Command_Leaderboard);

    RegAdminCmd("sm_removefromleaderboard", Command_RemoveFromLeaderboard, ADMFLAG_ROOT);
}

public APLRes AskPluginLoad2(Handle myself, bool late, char[] error, int err_max)
{
    CreateNative("SetDBData", Native_SetDBData);
    return APLRes_Success;
}

public void OnClientPutInServer(int client)
{
    if (g_dbScoreboard == null)
    {
        PrintToServer("No connection to database server");
        return;
    }

    if (!IsValidClient(client))
    {
        return;
    }

    g_bIsRanked[client] = false;
    g_fLastQuery[client] = 0.0;

    // Get client's SteamID
    char sSteamId[64];
    if (!GetClientAuthId(client, AuthId_Steam2, sSteamId, sizeof(sSteamId)))
    {
        PrintToServer("Couldn't fetch client's SteamID");
        return;
    }

    // Check if client is on leaderboard table
    char sLeaderboardQuery[256];
    Format(sLeaderboardQuery, sizeof(sLeaderboardQuery), "SELECT points FROM leaderboard WHERE steamid = '%s'", sSteamId);
    SQL_TQuery(g_dbScoreboard, Database_CheckLeaderboard, sLeaderboardQuery, GetClientUserId(client));
}

public void OnClientDisconnect(int client)
{
    g_bIsRanked[client] = false;
    g_iPoints[client] = 0;
    g_fLastQuery[client] = 0.0;
    g_iUnrankedKills[client] = 0;
}

public void OnClientSettingsChanged(int client)
{
    CreateTimer(5.0, Timer_UpdatePlayerName, GetClientUserId(client), TIMER_FLAG_NO_MAPCHANGE);
}

public Action Timer_UpdatePlayerName(Handle timer, int iUserId)
{
    int client = GetClientOfUserId(iUserId);

    if (g_dbScoreboard == null)
    {
        PrintToServer("No connection to database server");
        return Plugin_Stop;
    }

    if (!IsValidClient(client))
    {
        return Plugin_Stop;
    }

    // Get client's SteamID
    char sSteamId[64];
    if (!GetClientAuthId(client, AuthId_Steam2, sSteamId, sizeof(sSteamId)))
    {
        PrintToServer("Couldn't fetch client's SteamID");
        return Plugin_Stop;
    }

    // Get client's name and sanitize it for SQL (double apostrophes)
    char sClientName[64];
    GetClientName(client, sClientName, sizeof(sClientName));
    ReplaceString(sClientName, sizeof(sClientName), "'", "''");

    if (g_bIsRanked[client])
    {
        // Update name in leaderboard database
        char sUpdateQuery[256];
        Format(sUpdateQuery, sizeof(sUpdateQuery), "UPDATE leaderboard SET name = '%s' WHERE steamid = '%s'", sClientName, sSteamId);
        SQL_TQuery(g_dbScoreboard, Database_Generic, sUpdateQuery);
    }
    return Plugin_Continue;
}

public void Database_Connect(Database db, const char[] error, any data)
{
    if (db == null)
    {
        PrintToServer("Database error: %s", error);
        return;
    }

    g_dbScoreboard = db;

    PrintToServer("Successfully connected to database server");

    // Create leaderboard table if it doesn't exist
    SQL_TQuery(g_dbScoreboard, Database_Generic, "CREATE TABLE IF NOT EXISTS leaderboard (steamid VARCHAR(64) NOT NULL, name VARCHAR(64) NOT NULL, points INT NOT NULL, kills INT NOT NULL, deaths INT NOT NULL, duel_wins INT NOT NULL, duel_losses INT NOT NULL, PRIMARY KEY (steamid))");
}

public void Database_CheckLeaderboard(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        PrintToServer("Database_CheckLeaderboard returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    // Get client steamID
    char sSteamId[64];
    GetClientAuthId(client, AuthId_Steam2, sSteamId, sizeof(sSteamId));

    // Get client's name
    char sClientName[64];
    GetClientName(client, sClientName, sizeof(sClientName));

    if (SQL_FetchRow(results))
    {
        int iPointsField;
        SQL_FieldNameToNum(results, "points", iPointsField);
        g_iPoints[client] = SQL_FetchInt(results, iPointsField);

        // Sanitize
        ReplaceString(sClientName, sizeof(sClientName), "'", "''");

        // Update name in database
        char sUpdateQuery[256];
        Format(sUpdateQuery, sizeof(sUpdateQuery), "UPDATE leaderboard SET name = '%s' WHERE steamid = '%s'", sClientName, sSteamId);
        SQL_TQuery(g_dbScoreboard, Database_Generic, sUpdateQuery);

        // Check rank and print
        char sRankQuery[256];
        Format(sRankQuery, sizeof(sRankQuery), "SELECT row_num, points FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, points FROM leaderboard) leaderboard WHERE steamid = '%s'", sSteamId);
        SQL_TQuery(g_dbScoreboard, Database_CheckRank, sRankQuery, GetClientUserId(client));

        g_bIsRanked[client] = true;
    } else
    {
        g_bIsRanked[client] = false;
        g_iUnrankedKills[client] = 0;
        CPrintToChatAll("{yellow}[GFR] {default}Player: {lightgreen}%s {default}| {lightgreen}Unranked", sClientName);
    }
}

public void Database_CheckRank(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        PrintToServer("Database_CheckRank returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    if (SQL_FetchRow(results))
    {
        // Client's rank

        int rankField;
        SQL_FieldNameToNum(results, "row_num", rankField);
        int rank = SQL_FetchInt(results, rankField);

        int pointsField;
        SQL_FieldNameToNum(results, "points", pointsField);
        int points = SQL_FetchInt(results, pointsField);

        // Client's name
        char sClientName[64];
        GetClientName(client, sClientName, sizeof(sClientName));

        CPrintToChatAll("{yellow}[GFR] {default}Player: {lightgreen}%s {default}| Rank: {lightgreen}%d {default}| {lightgreen}%d {default}points", sClientName, rank, points);
    }
}

public void Database_CheckRankCommand(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        PrintToServer("Database_CheckRankCommand returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    if (SQL_FetchRow(results))
    {
        // Target's rank
        int rankField;
        SQL_FieldNameToNum(results, "row_num", rankField);
        int rank = SQL_FetchInt(results, rankField);

        // Target's name
        int nameField;
        SQL_FieldNameToNum(results, "name", nameField);
        char name[64]; 
        SQL_FetchString(results, nameField, name, sizeof(name));

        // Target's points
        int pointsField;
        SQL_FieldNameToNum(results, "points", pointsField);
        int points = SQL_FetchInt(results, pointsField);

        CPrintToChat(client, "{yellow}[GFR] {lightgreen}%s {default}is ranked {lightgreen}%d {default}with {lightgreen}%d {default}points", name, rank, points);
    } else
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Player is unranked");
    }
}

public void Database_CheckLeaderboardCommand(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        LogError("Database_CheckLeaderboardCommand returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    Menu menu = new Menu(Menu_LeaderboardCommand);
    menu.SetTitle("Leaderboard:");

    for (int i = 0; i < 100; i++)
    {
        if (SQL_FetchRow(results))
        {
            // Rank
            int rankField;
            SQL_FieldNameToNum(results, "row_num", rankField);
            int iRank = SQL_FetchInt(results, rankField);

            // Name
            int nameField;
            SQL_FieldNameToNum(results, "name", nameField);
            char sName[64];
            SQL_FetchString(results, nameField, sName, sizeof(sName));

            // SteamID
            int steamIdField;
            SQL_FieldNameToNum(results, "steamid", steamIdField);
            char sSteamId[64];
            SQL_FetchString(results, steamIdField, sSteamId, sizeof(sSteamId));

            char sPlayerName[128];
            Format(sPlayerName, sizeof(sPlayerName), "| Rank %d: %s", iRank, sName);

            menu.AddItem(sSteamId, sPlayerName);
        } else
        {
            break;
        }
    }

    menu.Display(client, 30);
}

public void Database_CheckInfo(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        LogError("Database_CheckInfo returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    if (SQL_FetchRow(results))
    {
        // Name
        int nameField;
        SQL_FieldNameToNum(results, "name", nameField);
        char sName[64];
        SQL_FetchString(results, nameField, sName, sizeof(sName));

        char sNameInfo[128];
        Format(sNameInfo, sizeof(sNameInfo), "Name: %s", sName);

        // Rank
        int rankField;
        SQL_FieldNameToNum(results, "row_num", rankField);
        int iRank = SQL_FetchInt(results, rankField);

        char sRankInfo[128];
        Format(sRankInfo, sizeof(sRankInfo), "Rank: %d", iRank);

        // SteamID
        int steamIdField;
        SQL_FieldNameToNum(results, "steamid", steamIdField);
        char sSteamId[64];
        SQL_FetchString(results, steamIdField, sSteamId, sizeof(sSteamId));

        char sSteamIdInfo[128];
        Format(sSteamIdInfo, sizeof(sSteamIdInfo), "SteamID: %s", sSteamId);

        // Points
        int pointsField;
        SQL_FieldNameToNum(results, "points", pointsField);
        int iPoints = SQL_FetchInt(results, pointsField);

        char sPointsInfo[128];
        Format(sPointsInfo, sizeof(sPointsInfo), "Points: %d", iPoints);

        // Kills
        int killsField;
        SQL_FieldNameToNum(results, "kills", killsField);
        int iKills = SQL_FetchInt(results, killsField);

        char sKillsInfo[128];
        Format(sKillsInfo, sizeof(sKillsInfo), "Kills: %d", iKills);

        // Deaths
        int deathsField;
        SQL_FieldNameToNum(results, "deaths", deathsField);
        int iDeaths = SQL_FetchInt(results, deathsField);

        char sDeathsInfo[128];
        Format(sDeathsInfo, sizeof(sDeathsInfo), "Deaths: %d", iDeaths);

        // KDR
        char sKDRInfo[128];
        float fKDR = (float)(iKills) / (float)(iDeaths);
        Format(sKDRInfo, sizeof(sKDRInfo), "KDR: %.2f", fKDR);

        // Duel wins
        int duelWinsField;
        SQL_FieldNameToNum(results, "duel_wins", duelWinsField);
        int iDuelWins = SQL_FetchInt(results, duelWinsField);

        char sDuelWinsInfo[128];
        Format(sDuelWinsInfo, sizeof(sDuelWinsInfo), "Duel wins: %d", iDuelWins);

        // Duel losses
        int duelLossesField;
        SQL_FieldNameToNum(results, "duel_losses", duelLossesField);
        int iDuelLosses = SQL_FetchInt(results, duelLossesField);

        char sDuelLossesInfo[128];
        Format(sDuelLossesInfo, sizeof(sDuelLossesInfo), "Duel losses: %d", iDuelLosses);

        // WLR
        char sWLRInfo[128];
        float fWLR = (float)(iDuelWins) / (float)(iDuelLosses);
        Format(sWLRInfo, sizeof(sWLRInfo), "WLR: %.2f\n ", fWLR);

        // Info panel
        Panel panel = new Panel();
        panel.SetTitle("Player Info:\n ");

        panel.DrawText(sNameInfo);
        panel.DrawText(sRankInfo);
        panel.DrawText(sSteamIdInfo);
        panel.DrawText(sPointsInfo);
        panel.DrawText(sKillsInfo);
        panel.DrawText(sDeathsInfo);
        panel.DrawText(sKDRInfo);
        panel.DrawText(sDuelWinsInfo);
        panel.DrawText(sDuelLossesInfo);
        panel.DrawText(sWLRInfo);

        panel.DrawText("0. Exit");

        panel.Send(client, Panel_Info, 30);
    }
}

public void Database_ShowKillMessage(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        LogError("Database_ShowKillMessage returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    if (SQL_FetchRow(results))
    {
        // Rank
        int rankField;
        SQL_FieldNameToNum(results, "row_num", rankField);
        int iRank = SQL_FetchInt(results, rankField);

        // Name
        int nameField;
        SQL_FieldNameToNum(results, "name", nameField);
        char sName[64];
        SQL_FetchString(results, nameField, sName, sizeof(sName));

        CPrintToChat(client, "{yellow}[GFR] {green}+2 {default}points for killing {lightgreen}%s {default}({lightgreen}#%d{default})", sName, iRank);
    }
}

public void Database_ShowDeathMessage(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        LogError("Database_ShowDeathMessage returned error: %s", error);
        return;
    }

    int client = GetClientOfUserId(data);

    if (!IsValidClient(client))
    {
        return;
    }

    if (SQL_FetchRow(results))
    {
        // Rank
        int rankField;
        SQL_FieldNameToNum(results, "row_num", rankField);
        int iRank = SQL_FetchInt(results, rankField);

        // Name
        int nameField;
        SQL_FieldNameToNum(results, "name", nameField);
        char sName[64];
        SQL_FetchString(results, nameField, sName, sizeof(sName));

        CPrintToChat(client, "{yellow}[GFR] {red}-2 {default}points for dying to {lightgreen}%s {default}({lightgreen}#%d{default})", sName, iRank);
    }
}

public void Database_Generic(Handle owner, Handle results, const char[] error, any data)
{
    if (owner == null || results == null)
    {
        LogError("Database_Generic returned error: %s", error);
        return;
    }
}

public Action Command_RemoveFromLeaderboard(int client, int args)
{
    if (g_dbScoreboard != null)
    {
        if (args == 5)
        {
            char sWholeSteamId[64];
            char sArgSteamId1[16];
            GetCmdArg(1, sArgSteamId1, sizeof(sArgSteamId1));
            char sArgSteamId2[8];
            GetCmdArg(2, sArgSteamId2, sizeof(sArgSteamId2));
            char sArgSteamId3[8];
            GetCmdArg(3, sArgSteamId3, sizeof(sArgSteamId3));
            char sArgSteamId4[8];
            GetCmdArg(4, sArgSteamId4, sizeof(sArgSteamId4));
            char sArgSteamId5[16];
            GetCmdArg(5, sArgSteamId5, sizeof(sArgSteamId5));

            Format(sWholeSteamId, sizeof(sWholeSteamId), "%s%s%s%s%s", sArgSteamId1, sArgSteamId2, sArgSteamId3, sArgSteamId4, sArgSteamId5);

            if (StrContains(sArgSteamId1, "STEAM_", true) != -1)
            {
                char sRemoveQuery[256];
                Format(sRemoveQuery, sizeof(sRemoveQuery), "DELETE FROM leaderboard WHERE steamid = '%s'", sWholeSteamId);
                SQL_TQuery(g_dbScoreboard, Database_Generic, sRemoveQuery);
                CPrintToChat(client, "{yellow}[GFR] {default}Removed {lightgreen}%s {default}from leaderboard", sWholeSteamId);
            } else
            {
                CPrintToChat(client, "{yellow}[GFR] {default}Not a proper steamID (E.g. STEAM_0:0:00000000)");
            }
        } else
        {
            CPrintToChat(client, "{yellow}[GFR] {default}sm_removefromleaderboard <steamID>");
        }
    } else 
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Server is not connected to the database");
    }
    return Plugin_Handled;
}

public Action Command_Leaderboard(int client, int args)
{
    if (g_dbScoreboard != null)
    {
        if (g_fLastQuery[client] + 5 <= GetTickedTime())
        {
            char sLeaderboardQuery[256];
            Format(sLeaderboardQuery, sizeof(sLeaderboardQuery), "SELECT row_num, steamid, name FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name FROM leaderboard) leaderboard LIMIT 100");
            SQL_TQuery(g_dbScoreboard, Database_CheckLeaderboardCommand, sLeaderboardQuery, GetClientUserId(client));
            g_fLastQuery[client] = GetTickedTime();
        } else
        {
            CPrintToChat(client, "{yellow}[GFR] {default}You have to wait to use this command again");
        }
    } else
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Server is not connected to the database");
    }
    return Plugin_Handled;
}

public Action Command_Rank(int client, int args)
{
    if (g_dbScoreboard != null)
    {
        if (g_fLastQuery[client] + 5 <= GetTickedTime())
        {
            if (args == 1)
            {
                // Check through clients to find a player with the name
                char sArgName[64];
                GetCmdArg(1, sArgName, sizeof(sArgName));
                for (int i = 1; i <= MaxClients; i++)
                {
                    if (IsClientInGame(i))
                    {
                        char sTargetClientName[64];
                        GetClientName(i, sTargetClientName, sizeof(sTargetClientName));
                        if (StrContains(sTargetClientName, sArgName, false) != -1)
                        {
                            // If name found
                            if (g_bIsRanked[i])
                            {
                                char sTargetSteamId[64];
                                GetClientAuthId(i, AuthId_Steam2, sTargetSteamId, sizeof(sTargetSteamId));

                                char sRankQuery[512];
                                Format(sRankQuery, sizeof(sRankQuery), "SELECT row_num, name, points FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name, points FROM leaderboard) leaderboard WHERE steamid = '%s'", sTargetSteamId);
                                SQL_TQuery(g_dbScoreboard, Database_CheckRankCommand, sRankQuery, GetClientUserId(client));
                            } else
                            {
                                CPrintToChat(client, "{yellow}[GFR] {lightgreen}%s {default}is unranked", sTargetClientName);
                            }
                            g_fLastQuery[client] = GetTickedTime();
                            return Plugin_Handled;
                        }
                    }
                }
                CPrintToChat(client, "{yellow}[GFR] {default}Could not find player");
            } else
            {
                char sOwnSteamId[64];
                GetClientAuthId(client, AuthId_Steam2, sOwnSteamId, sizeof(sOwnSteamId));

                char sRankQuery[512];
                Format(sRankQuery, sizeof(sRankQuery), "SELECT row_num, name, points FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name, points FROM leaderboard) leaderboard WHERE steamid = '%s'", sOwnSteamId);
                SQL_TQuery(g_dbScoreboard, Database_CheckRankCommand, sRankQuery, GetClientUserId(client));
            }
            g_fLastQuery[client] = GetTickedTime();
        } else
        {
            CPrintToChat(client, "{yellow}[GFR] {default}You have to wait to use this command again");
        }
    } else
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Server is not connected to the database");
    }
    return Plugin_Handled;
}

public int Panel_Info(Menu menu, MenuAction action, int param1, int param2)
{
    if (action == MenuAction_End)
    {
        delete menu;
    }
    return 0;
}

public int Menu_LeaderboardCommand(Menu menu, MenuAction action, int param1, int param2)
{
    switch (action)
    {
        case MenuAction_Select:
        {
            char item[64];
            menu.GetItem(param2, item, sizeof(item));

            char sInfoQuery[512];
            Format(sInfoQuery, sizeof(sInfoQuery), "SELECT row_num, steamid, name, points, kills, deaths, duel_wins, duel_losses FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name, points, kills, deaths, duel_wins, duel_losses FROM leaderboard) leaderboard WHERE steamid = '%s'", item);
            SQL_TQuery(g_dbScoreboard, Database_CheckInfo, sInfoQuery, GetClientUserId(param1));
        }
        case MenuAction_End:
        {
            delete menu;
        }
    }
    return 0;
}

public void Event_PlayerDeath(Event event, const char[] name, bool bDontBroadcast)
{
    if (g_dbScoreboard == null)
    {
        return;
    }

    int iVictimId = GetEventInt(event, "userid");
    int iAttackerId = GetEventInt(event, "attacker");

    if (!iAttackerId || iVictimId == iAttackerId)
    {
        return;
    }

    // Get victim steamID
    int iVictimClient = GetClientOfUserId(iVictimId);

    if (!IsValidClient(iVictimClient))
    {
        return;
    }

    char sVictimSteamId[64];
    GetClientAuthId(iVictimClient, AuthId_Steam2, sVictimSteamId, sizeof(sVictimSteamId));

    // Get attacker steamID
    int iAttackerClient = GetClientOfUserId(iAttackerId);

    if (!IsValidClient(iAttackerClient))
    {
        return;
    }

    char sAttackerSteamId[64];
    GetClientAuthId(iAttackerClient, AuthId_Steam2, sAttackerSteamId, sizeof(sAttackerSteamId));

    if (g_bIsRanked[iVictimClient])
    {
        char sUpdateQuery[256];
        if (g_iPoints[iVictimClient] <= 0)
        {
            Format(sUpdateQuery, sizeof(sUpdateQuery), "UPDATE leaderboard SET deaths = deaths + 1 WHERE steamid = '%s'", sVictimSteamId);
        } else
        {
            g_iPoints[iVictimClient] -= 2;
            Format(sUpdateQuery, sizeof(sUpdateQuery), "UPDATE leaderboard SET deaths = deaths + 1, points = points - 2 WHERE steamid = '%s'", sVictimSteamId);
            if (g_bIsRanked[iAttackerClient])
            {
                char sDeathMsgQuery[256];
                Format(sDeathMsgQuery, sizeof(sDeathMsgQuery), "SELECT row_num, name FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name FROM leaderboard) leaderboard WHERE steamid = '%s'", sAttackerSteamId);
                SQL_TQuery(g_dbScoreboard, Database_ShowDeathMessage, sDeathMsgQuery, iVictimId);
            } else 
            {
                char sAttackerName[64];
                GetClientName(iAttackerClient, sAttackerName, sizeof(sAttackerName));
                CPrintToChat(iVictimClient, "{yellow}[GFR] {red}-2 {default}points for dying to {lightgreen}%s {default}({lightgreen}Unranked{default})", sAttackerName);
            }
        }
        SQL_TQuery(g_dbScoreboard, Database_Generic, sUpdateQuery);
    }

    if (g_bIsRanked[iAttackerClient])
    {
        g_iPoints[iAttackerClient] += 2;
        char sUpdateQuery[256];
        Format(sUpdateQuery, sizeof(sUpdateQuery), "UPDATE leaderboard SET kills = kills + 1, points = points + 2 WHERE steamid = '%s'", sAttackerSteamId);
        SQL_TQuery(g_dbScoreboard, Database_Generic, sUpdateQuery);
        if (g_bIsRanked[iVictimClient])
        {
            char sKillMsgQuery[256];
            Format(sKillMsgQuery, sizeof(sKillMsgQuery), "SELECT row_num, name FROM (SELECT ROW_NUMBER() OVER (ORDER BY points DESC) row_num, steamid, name FROM leaderboard) leaderboard WHERE steamid = '%s'", sVictimSteamId);
            SQL_TQuery(g_dbScoreboard, Database_ShowKillMessage, sKillMsgQuery, iAttackerId);
        } else
        {
            char sVictimName[64];
            GetClientName(iVictimClient, sVictimName, sizeof(sVictimName));
            CPrintToChat(iAttackerClient, "{yellow}[GFR] {green}+2 {default}points for killing {lightgreen}%s {default}({lightgreen}Unranked{default})", sVictimName);
        }
    } else
    {
        g_iUnrankedKills[iAttackerClient]++;
        CPrintToChat(iAttackerClient, "{yellow}[GFR] {lightgreen}%d{default}/{lightgreen}20 {default}kills to get be ranked", g_iUnrankedKills[iAttackerClient]);

        // When unranked player gets 20 kills place them on the leaderboard
        if (g_iUnrankedKills[iAttackerClient] >= 20)
        {
            g_bIsRanked[iAttackerClient] = true;

            // Get client's name and sanitize it for SQL (double apostrophes)
            char sAttackerClientName[64];
            GetClientName(iAttackerClient, sAttackerClientName, sizeof(sAttackerClientName));
            ReplaceString(sAttackerClientName, sizeof(sAttackerClientName), "'", "''");

            g_iPoints[iAttackerClient] = 100;

            char sInsertQuery[256];
            Format(sInsertQuery, sizeof(sInsertQuery), "INSERT INTO leaderboard (steamid, name, points, kills, deaths, duel_wins, duel_losses) VALUES ('%s', '%s', 100, 0, 0, 0, 0)", sAttackerSteamId, sAttackerClientName);
            SQL_TQuery(g_dbScoreboard, Database_Generic, sInsertQuery, GetClientUserId(iAttackerClient));
        }
    }
}

public int Native_SetDBData(Handle plugin, int numParams)
{
    int iWinnerUserId = GetNativeCell(1);
    int iLoserUserId = GetNativeCell(2);

    int iWinnerClient = GetClientOfUserId(iWinnerUserId);
    int iLoserClient = GetClientOfUserId(iLoserUserId);

    if (!IsValidClient(iWinnerClient))
    {
        return 0;
    }
    
    if (!IsValidClient(iLoserClient))
    {
        return 0;
    }

    int iWinnerIdLength;
    GetNativeStringLength(3, iWinnerIdLength);

    if (iWinnerIdLength <= 0)
    {
        return 0;
    }

    int iLoserIdLength;
    GetNativeStringLength(4, iLoserIdLength);

    if (iLoserIdLength <= 0)
    {
        return 0;
    }

    // Duel score

    if (g_bIsRanked[iWinnerClient])
    {
        char sWinnerSteamId[64];
        GetNativeString(3, sWinnerSteamId, 64);

        char sWinnerUpdateQuery[256];
        Format(sWinnerUpdateQuery, 256, "UPDATE leaderboard SET duel_wins = duel_wins + 1 WHERE steamid = '%s'", sWinnerSteamId);
        SQL_TQuery(g_dbScoreboard, Database_Generic, sWinnerUpdateQuery);
    }

    char sLoserSteamId[64];
    GetNativeString(4, sLoserSteamId, 64);

    if (g_bIsRanked[iLoserClient])
    {
        char sLoserUpdateQuery[256];
        Format(sLoserUpdateQuery, 256, "UPDATE leaderboard SET duel_losses = duel_losses + 1 WHERE steamid = '%s'", sLoserSteamId);
        SQL_TQuery(g_dbScoreboard, Database_Generic, sLoserUpdateQuery);
    }

    return 0;
}

bool IsValidClient(int client)
{
    if (0 < client && client <= MaxClients)
    {
        if (IsClientInGame(client))
        {
            return true;
        }
    }
    return false; 
}