#include <sourcemod>
#include <tf2_stocks>
#include <dueldbnatives>
#include <morecolors>

#pragma newdecls required
#pragma semicolon 1

#define SPECMODE_FIRSTPERSON 4
#define SPECMODE_3RDPERSON 5

public Plugin myinfo =
{
    name = "Dueling plugin",
    author = "SpookyMonster",
    description = "Plugin for dueling",
    version = "1.0",
    url = ""
};

enum struct DuelData
{
    char sDuelChallengerName[64];
    char sDuelChallengerSteamId[64];
    int iDuelChallengerUserId;
    int iDuelScore;
    int iDuelKillstreak;
    int iDuelTimerSecs;
    int iDuelTimerMins;
    bool bDuelTimeOver;
    int iDuelArena;
}

DuelData g_duel[MAXPLAYERS + 1];

// Duel arenas

float g_fRedSpawnAngle[] = { 0.0, 135.0, 0.0 };
float g_fBlueSpawnAngle[] = { 0.0, 315.0, 0.0 };

// Arena 1
float g_fRedSpawnOrigin1[] = { -7984.0, 2768.0, 1270.0 };
float g_fBlueSpawnOrigin1[] = { -11536.0, 6320.0, 1270.0 };
bool g_bReserved1 = false;

// Arena 2
float g_fRedSpawnOrigin2[] = { -3440.0, 2768.0, 1270.0 };
float g_fBlueSpawnOrigin2[] = { -6992.0, 6320.0, 1270.0 };
bool g_bReserved2 = false;

// Arena 3
float g_fRedSpawnOrigin3[] = { -7984.0, -1776.0, 1270.0 };
float g_fBlueSpawnOrigin3[] = { -11536.0, 1776.0, 1270.0 };
bool g_bReserved3 = false;

// Arena 4
float g_fRedSpawnOrigin4[] = { -3440.0, -1776.0, 1270.0 };
float g_fBlueSpawnOrigin4[] = { -6992.0, 1776.0, 1270.0 };
bool g_bReserved4 = false;

// Arena 5
float g_fRedSpawnOrigin5[] = { -7984.0, -6320.0, 1270.0 };
float g_fBlueSpawnOrigin5[] = { -11536.0, -2768.0, 1270.0 };
bool g_bReserved5 = false;

// Arena 6
float g_fRedSpawnOrigin6[] = { -3440.0, -6320.0, 1270.0 };
float g_fBlueSpawnOrigin6[] = { -6992.0, -2768.0, 1270.0 };
bool g_bReserved6 = false;

// Has client been challenged to a duel
bool g_bChallenged[MAXPLAYERS + 1];

// Has client challenged someone to a duel
bool g_bHasChallenged[MAXPLAYERS + 1];

// Is client in a duel
bool g_bInDuel[MAXPLAYERS + 1];

// Dueling cooldown
float g_fDuelCooldown[MAXPLAYERS + 1];

// Handle for timers for when you challenge a player
Handle g_hChallengeTimers[MAXPLAYERS + 1];

// Handle for duel start
Handle g_hDuelStartTimers[MAXPLAYERS + 1];

// Handle for duel clock timer
Handle g_hDuelClockTimer[MAXPLAYERS + 1];

// Handle for the timer for refreshing the duel score text
Handle g_hDuelScoreRefresh[MAXPLAYERS + 1];

// Handle for HUD text sync
Handle g_hHudSync;

public void OnPluginStart()
{
    AddCommandListener(CmdListener_Suicide, "kill");
    AddCommandListener(CmdListener_Suicide, "explode");

    HookEvent("player_death", Event_PlayerDeath, EventHookMode_Post);
    HookEvent("player_spawn", Event_PlayerSpawn);
    HookEvent("player_team", Event_ChangeTeam);
    HookEvent("player_changeclass", Event_ChangeClass);

    RegConsoleCmd("sm_duel", Command_Duel);

    RegConsoleCmd("sm_accept", Command_AcceptDuel);
    RegConsoleCmd("sm_decline", Command_DeclineDuel);

    RegConsoleCmd("sm_leave", Command_ResignDuel);
    RegConsoleCmd("sm_exit", Command_ResignDuel);
    RegConsoleCmd("sm_resign", Command_ResignDuel);
    RegConsoleCmd("sm_r", Command_ResignDuel);

    g_hHudSync = CreateHudSynchronizer();

    SetHudTextParams(0.02, 0.02, 1.0, 255, 255, 255, 255, 2, 0.0, 0.0, 0.0);
}

public void OnClientDisconnect(int client)
{
    // Someone in a duel disconnects
    if (g_bInDuel[client])
    {
        int iChallengeeClient = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);
        
        FreeArena(g_duel[client].iDuelArena);

        if (!IsValidClient(iChallengeeClient))
        {
            return;
        }

        // Delete timers
        delete g_hChallengeTimers[iChallengeeClient];
        delete g_hDuelStartTimers[iChallengeeClient];
        delete g_hDuelClockTimer[iChallengeeClient];
        delete g_hDuelScoreRefresh[iChallengeeClient];

        // Reset globals
        g_bInDuel[iChallengeeClient] = false;
        g_bChallenged[iChallengeeClient] = false;
        g_bHasChallenged[iChallengeeClient] = false;
        g_fDuelCooldown[iChallengeeClient] = GetTickedTime();

        // Reset duel data
        g_duel[iChallengeeClient].sDuelChallengerName = "";
        g_duel[iChallengeeClient].sDuelChallengerSteamId = "";
        g_duel[iChallengeeClient].iDuelChallengerUserId = 0;
        g_duel[iChallengeeClient].iDuelKillstreak = 0;
        g_duel[iChallengeeClient].iDuelTimerSecs = 0;
        g_duel[iChallengeeClient].iDuelTimerMins = 0;
        g_duel[iChallengeeClient].bDuelTimeOver = false;
        g_duel[iChallengeeClient].iDuelArena = 0;

        TF2_RespawnPlayer(iChallengeeClient);

        char sWinnerName[64];
        GetClientName(iChallengeeClient, sWinnerName, sizeof(sWinnerName));

        char sLoserName[64];
        GetClientName(client, sLoserName, sizeof(sLoserName));

        ClientCommand(iChallengeeClient, "playgamesound ui/duel_event.wav");

        CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}disconnected while in a duel with {lightgreen}%s", sLoserName, sWinnerName);
    }

    // Someone challenged to duel disconnects
    if (g_bChallenged[client])
    {
        int iChallengerClient = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        FreeArena(g_duel[client].iDuelArena);

        if (!IsValidClient(iChallengerClient))
        {
            return;
        }

        delete g_hChallengeTimers[iChallengerClient];
        delete g_hDuelStartTimers[iChallengerClient];
        delete g_hDuelClockTimer[iChallengerClient];
        delete g_hDuelScoreRefresh[iChallengerClient];

        g_bHasChallenged[iChallengerClient] = false;

        g_duel[iChallengerClient].iDuelArena = 0;

        // Play sound effect

        ClientCommand(iChallengerClient, "playgamesound ui/duel_challenge_rejected.wav");

        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}The player you challenged has disconnected");
    }

    // Someone who has challenged someone disconnects
    if (g_bHasChallenged[client])
    {
        int iChallengeeClient = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        FreeArena(g_duel[client].iDuelArena);

        if (!IsValidClient(iChallengeeClient))
        {
            return;
        }

        delete g_hChallengeTimers[iChallengeeClient];
        delete g_hDuelStartTimers[iChallengeeClient];
        delete g_hDuelClockTimer[iChallengeeClient];
        delete g_hDuelScoreRefresh[iChallengeeClient];

        g_duel[iChallengeeClient].iDuelArena = 0;

        g_bChallenged[iChallengeeClient] = false;

        CPrintToChat(iChallengeeClient, "{yellow}[GFR] {default}Your challenger has disconnected");
    }

    // Reset globals
    g_bInDuel[client] = false;
    g_bChallenged[client] = false;
    g_bHasChallenged[client] = false;
    g_fDuelCooldown[client] = 0.0;

    // Kill timers
    delete g_hChallengeTimers[client];
    delete g_hDuelStartTimers[client];
    delete g_hDuelClockTimer[client];
    delete g_hDuelScoreRefresh[client];

    // Duel data
    g_duel[client].sDuelChallengerName = "";
    g_duel[client].sDuelChallengerSteamId = "";
    g_duel[client].iDuelChallengerUserId = 0;
    g_duel[client].iDuelScore = 0;
    g_duel[client].iDuelKillstreak = 0;
    g_duel[client].iDuelTimerSecs = 0;
    g_duel[client].iDuelTimerMins = 0;
    g_duel[client].bDuelTimeOver = false;
    g_duel[client].iDuelArena = 0;
}

public void Event_PlayerDeath(Event event, const char[] name, bool bDontBroadcast)
{
    int iVictimId = GetEventInt(event, "userid");
    int iAttackerId = GetEventInt(event, "attacker");

    int iVictimClient = GetClientOfUserId(iVictimId);

    if (!IsValidClient(iVictimClient))
    {
        return;
    }

    int iAttackerClient = GetClientOfUserId(iAttackerId);

    if (!IsValidClient(iAttackerClient))
    {
        return;
    }

    if (g_bInDuel[iVictimClient] && g_duel[iVictimClient].iDuelChallengerUserId == iAttackerId)
    {
        g_duel[iAttackerClient].iDuelScore++;
        g_duel[iAttackerClient].iDuelKillstreak++;

        g_duel[iVictimClient].iDuelKillstreak = 0;

        if (g_duel[iVictimClient].bDuelTimeOver)
        {
            // Show final score
            ShowScoreSpecDueler(iAttackerClient, iVictimClient);
            ShowScoreSpecDueler(iVictimClient, iAttackerClient);

            // Win after time over
            DuelEnd(iAttackerClient, iVictimClient);

            char sWinnerName[64];
            GetClientName(iAttackerClient, sWinnerName, sizeof(sWinnerName));
            char sLoserName[64];
            GetClientName(iVictimClient, sLoserName, sizeof(sLoserName));
            
            CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}after golden point. Score: {lightgreen}%d{default}-{lightgreen}%d", sWinnerName, sLoserName, g_duel[iAttackerClient].iDuelScore, g_duel[iVictimClient].iDuelScore);
            return;
        }

        if (g_duel[iAttackerClient].iDuelScore == 10)
        {
            // Show final score
            ShowScoreSpecDueler(iAttackerClient, iVictimClient);
            ShowScoreSpecDueler(iVictimClient, iAttackerClient);

            // Win with 10 kills
            DuelEnd(iAttackerClient, iVictimClient);

            char sWinnerName[64];
            GetClientName(iAttackerClient, sWinnerName, sizeof(sWinnerName));
            char sLoserName[64];
            GetClientName(iVictimClient, sLoserName, sizeof(sLoserName));
            
            CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}with 10 kills. Score: {lightgreen}%d{default}-{lightgreen}%d", sWinnerName, sLoserName, g_duel[iAttackerClient].iDuelScore, g_duel[iVictimClient].iDuelScore);
            return;
        }

        if (g_duel[iAttackerClient].iDuelKillstreak == 4)
        {
            // Show final score
            ShowScoreSpecDueler(iAttackerClient, iVictimClient);
            ShowScoreSpecDueler(iVictimClient, iAttackerClient);

            // Win with domination
            DuelEnd(iAttackerClient, iVictimClient);

            char sWinnerName[64];
            GetClientName(iAttackerClient, sWinnerName, sizeof(sWinnerName));
            char sLoserName[64];
            GetClientName(iVictimClient, sLoserName, sizeof(sLoserName));

            CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}with a domination. Score: {lightgreen}%d{default}-{lightgreen}%d", sWinnerName, sLoserName, g_duel[iAttackerClient].iDuelScore, g_duel[iVictimClient].iDuelScore);
            return;
        }
    }
}

public void Event_ChangeClass(Event hEvent, const char[] strEventName, bool bDontBroadcast)
{
    int client = GetClientOfUserId(GetEventInt(hEvent, "userid"));

    if (!IsValidClient(client))
    {
        return;
    }

    if (g_bInDuel[client])
    {
        int challenger = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        DuelEnd(challenger, client);

        char sWinnerName[64];
        GetClientName(challenger, sWinnerName, sizeof(sWinnerName));
        char sLoserName[64];
        GetClientName(client, sLoserName, sizeof(sLoserName));

        CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}because they resigned", sWinnerName, sLoserName);
    }
}

public void Event_ChangeTeam(Event hEvent, const char[] strEventName, bool bDontBroadcast)
{
    int client = GetClientOfUserId(GetEventInt(hEvent, "userid"));

    if (!IsValidClient(client))
    {
        return;
    }

    if (g_bInDuel[client])
    {
        // Lose match
        int challenger = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        DuelEnd(challenger, client);

        char sWinnerName[64];
        GetClientName(challenger, sWinnerName, sizeof(sWinnerName));
        char sLoserName[64];
        GetClientName(client, sLoserName, sizeof(sLoserName));

        CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}because they resigned", sWinnerName, sLoserName);
    }
}

public Action Event_PlayerSpawn(Handle hEvent, char[] strEventName, bool bDontBroadcast)
{
    int client = GetClientOfUserId(GetEventInt(hEvent, "userid"));

    if (!IsValidClient(client))
    {
        return Plugin_Handled;
    }

    if (g_bInDuel[client])
    {
        TeleportPlayerToArena(GetClientUserId(client));
    }

    return Plugin_Continue;
}

public Action Command_Duel(int client, int args)
{
    if (g_bInDuel[client])
    {
        CPrintToChat(client, "{yellow}[GFR] {default}You are already in a duel");
        return Plugin_Handled;
    }
    
    if (g_bChallenged[client])
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Respond to the duel challenge first");
        return Plugin_Handled;
    }

    if (g_bHasChallenged[client])
    {
        CPrintToChat(client, "{yellow}[GFR] {default}You have already sent a challenge");
        return Plugin_Handled;
    }

    if (g_fDuelCooldown[client] + 30.0 <= GetTickedTime())
    {
        // Args is name, if no args open menu with players
        if (args == 0)
        {
            Menu menu = new Menu(Menu_DuelPlayers);
            menu.SetTitle("Duel player:");

            for (int i = 1; i <= MaxClients; i++)
            {
                if (i != client)
                {
                    if (IsClientInGame(i))
                    {
                        int iTargetUserId = GetClientUserId(i);

                        char sTargetUserId[8];
                        IntToString(iTargetUserId, sTargetUserId, sizeof(sTargetUserId));

                        char sTargetName[64];
                        GetClientName(i, sTargetName, sizeof(sTargetName));

                        menu.AddItem(sTargetUserId, sTargetName);
                    }
                }
            }

            menu.Display(client, 30);
        } else
        {
            char sArgName[64];
            GetCmdArg(1, sArgName, sizeof(sArgName));

            for (int i = 1; i <= MaxClients; i++)
            {
                if (i != client)
                {
                    if (IsClientInGame(i))
                    {
                        char sTargetName[64];
                        GetClientName(i, sTargetName, sizeof(sTargetName));
                        if (StrContains(sTargetName, sArgName, false) != -1)
                        {
                            SendDuelChallenge(GetClientUserId(client), GetClientUserId(i));
                            return Plugin_Handled;
                        }
                    }
                }
            }
            CPrintToChat(client, "{yellow}[GFR] {default}Could not find player");
        }
    } else
    {
        int iWaitTime = RoundFloat(30 - GetTickedTime() + g_fDuelCooldown[client]);
        CPrintToChat(client, "{yellow}[GFR] {default}You have to wait for {lightgreen}%d {default}seconds to challenge someone again", iWaitTime);
    }
    return Plugin_Handled;
}

public int Menu_DuelPlayers(Menu menu, MenuAction action, int param1, int param2)
{
    switch (action)
    {
        case MenuAction_Select:
        {
            char item[8];
            menu.GetItem(param2, item, sizeof(item));

            int iTargetUserId = StringToInt(item);

            SendDuelChallenge(GetClientUserId(param1), iTargetUserId);
        }
        case MenuAction_End:
        {
            delete menu;
        }
    }
    return 0;
}

public Action Command_AcceptDuel(int client, int args)
{
    if (g_bChallenged[client])
    {
        int iChallengerUserId = g_duel[client].iDuelChallengerUserId;
        int iChallengerClient = GetClientOfUserId(iChallengerUserId);

        delete g_hChallengeTimers[iChallengerClient];

        if (!IsValidClient(iChallengerClient))
        {
            CPrintToChat(client, "{yellow}[GFR] {default}Challenger disconnected");
            g_bChallenged[client] = false;
            return Plugin_Handled;
        }

        char sChallengerName[64];
        GetClientName(iChallengerClient, sChallengerName, sizeof(sChallengerName));

        char sChallengeeName[64];
        GetClientName(client, sChallengeeName, sizeof(sChallengeeName));

        CPrintToChat(client, "{yellow}[GFR] {default}You accepted the challenge sent by {lightgreen}%s. {default}Your duel will start in 10 seconds", sChallengerName);
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {lightgreen}%s {default}accepted your challenge. Your duel will start in 10 seconds", sChallengeeName);

        // Play sound

        ClientCommand(iChallengerClient, "playgamesound ui/duel_challenge_accepted.wav");

        g_hDuelStartTimers[iChallengerClient] = CreateTimer(10.0, Timer_DuelStart, iChallengerUserId, TIMER_FLAG_NO_MAPCHANGE);
    } else
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Nothing to accept");
    }
    return Plugin_Handled;
}

public Action Command_DeclineDuel(int client, int args)
{
    if (g_bChallenged[client])
    {
        int iChallengerClient = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        delete g_hChallengeTimers[iChallengerClient];

        FreeArena(g_duel[client].iDuelArena);

        if (!IsValidClient(iChallengerClient))
        {
            CPrintToChat(client, "{yellow}[GFR] {default}Challenger disconnected");
            g_bChallenged[client] = false;
            return Plugin_Handled;
        }

        g_duel[client].iDuelArena = 0;
        g_duel[iChallengerClient].iDuelArena = 0;

        char sChallengerName[64];
        GetClientName(iChallengerClient, sChallengerName, sizeof(sChallengerName));

        char sChallengeeName[64];
        GetClientName(client, sChallengeeName, sizeof(sChallengeeName));

        CPrintToChat(client, "{yellow}[GFR] {default}You have declined the duel challenge sent by {lightgreen}%s", sChallengerName);
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {lightgreen}%s {default}has declined your duel challenge", sChallengeeName);

        // Play sound

        ClientCommand(iChallengerClient, "playgamesound ui/duel_challenge_rejected.wav");

        g_duel[client].iDuelChallengerUserId = 0;
        g_duel[iChallengerClient].iDuelChallengerUserId = 0;

        g_bChallenged[client] = false;
        g_bHasChallenged[iChallengerClient] = false;

        g_fDuelCooldown[iChallengerClient] = GetTickedTime();
    } else
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Nothing to decline");
    }
    return Plugin_Handled;
}

public Action Command_ResignDuel(int client, int args)
{
    if (g_bInDuel[client])
    {
        // Resign
        int challenger = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

        DuelEnd(challenger, client);

        char sWinnerName[64];
        GetClientName(challenger, sWinnerName, sizeof(sWinnerName));
        char sLoserName[64];
        GetClientName(client, sLoserName, sizeof(sLoserName));

        CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}because they resigned", sWinnerName, sLoserName);
    } else
    {
        TF2_RespawnPlayer(client);
    }
    return Plugin_Handled;
}

void DuelEnd(int iWinnerClient, int iLoserClient)
{
    SetDBData(GetClientUserId(iWinnerClient), GetClientUserId(iLoserClient), g_duel[iLoserClient].sDuelChallengerSteamId, g_duel[iWinnerClient].sDuelChallengerSteamId);

    ClientCommand(iWinnerClient, "playgamesound ui/duel_event.wav");
    ClientCommand(iLoserClient, "playgamesound ui/duel_score_behind.wav");

    // Delete timers
    delete g_hChallengeTimers[iWinnerClient];
    delete g_hDuelStartTimers[iWinnerClient];
    delete g_hDuelClockTimer[iWinnerClient];
    delete g_hDuelScoreRefresh[iWinnerClient];

    delete g_hChallengeTimers[iLoserClient];
    delete g_hDuelStartTimers[iLoserClient];
    delete g_hDuelClockTimer[iLoserClient];
    delete g_hDuelScoreRefresh[iLoserClient];

    // Reset globals
    g_bInDuel[iWinnerClient] = false;
    g_bChallenged[iWinnerClient] = false;
    g_bHasChallenged[iWinnerClient] = false;
    g_fDuelCooldown[iWinnerClient] = GetTickedTime();

    g_bInDuel[iLoserClient] = false;
    g_bChallenged[iLoserClient] = false;
    g_bHasChallenged[iLoserClient] = false;
    g_fDuelCooldown[iLoserClient] = GetTickedTime();

    // Free arena
    FreeArena(g_duel[iWinnerClient].iDuelArena);
    g_duel[iWinnerClient].iDuelArena = 0;
    g_duel[iLoserClient].iDuelArena = 0;

    // Reset duel data
    g_duel[iWinnerClient].sDuelChallengerName = "";
    g_duel[iWinnerClient].sDuelChallengerSteamId = "";
    g_duel[iWinnerClient].iDuelChallengerUserId = 0;
    g_duel[iWinnerClient].iDuelKillstreak = 0;
    g_duel[iWinnerClient].iDuelTimerSecs = 0;
    g_duel[iWinnerClient].iDuelTimerMins = 0;
    g_duel[iWinnerClient].bDuelTimeOver = false;

    g_duel[iLoserClient].sDuelChallengerName = "";
    g_duel[iLoserClient].sDuelChallengerSteamId = "";
    g_duel[iLoserClient].iDuelChallengerUserId = 0;
    g_duel[iLoserClient].iDuelKillstreak = 0;
    g_duel[iLoserClient].iDuelTimerSecs = 0;
    g_duel[iLoserClient].iDuelTimerMins = 0;
    g_duel[iLoserClient].bDuelTimeOver = false;

    if (IsValidClient(iWinnerClient))
    {
        TF2_RespawnPlayer(iWinnerClient);
    }

    if (IsValidClient(iLoserClient))
    {
        TF2_RespawnPlayer(iLoserClient);
    }
}

void SendDuelChallenge(int iChallengerUserId, int iChallengeeUserId)
{
    int iChallengerClient = GetClientOfUserId(iChallengerUserId);

    if (!IsValidClient(iChallengerClient))
    {
        return;
    }

    if (g_bChallenged[iChallengerClient])
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}Respond to the duel challenge first");
        return;
    }

    if (g_bHasChallenged[iChallengerClient])
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}You have already sent a challenge");
        return;
    }

    if (g_bInDuel[iChallengerClient])
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}You are already in a duel");
        return;
    }

    int iChallengeeClient = GetClientOfUserId(iChallengeeUserId);

    if (!IsValidClient(iChallengeeClient))
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}Player disconnected");
        return;
    }

    char sChallengeeName[64];
    GetClientName(iChallengeeClient, sChallengeeName, sizeof(sChallengeeName));

    if (g_bChallenged[iChallengeeClient] || g_bHasChallenged[iChallengeeClient])
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {lightgreen}%s {default}has already been challenged", sChallengeeName);
        return;
    }

    if (g_bInDuel[iChallengeeClient])
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {lightgreen}%s {default}is already in a duel", sChallengeeName);
        return;
    }

    // Reserve arena

    if (!g_bReserved1)
    {
        g_duel[iChallengerClient].iDuelArena = 1;
        g_duel[iChallengeeClient].iDuelArena = 1;
        g_bReserved1 = true;
    } else if (!g_bReserved2)
    {
        g_duel[iChallengerClient].iDuelArena = 2;
        g_duel[iChallengeeClient].iDuelArena = 2;
        g_bReserved2 = true;
    } else if (!g_bReserved3)
    {
        g_duel[iChallengerClient].iDuelArena = 3;
        g_duel[iChallengeeClient].iDuelArena = 3;
        g_bReserved3 = true;
    } else if (!g_bReserved4)
    {
        g_duel[iChallengerClient].iDuelArena = 4;
        g_duel[iChallengeeClient].iDuelArena = 4;
        g_bReserved4 = true;
    } else if (!g_bReserved5)
    {
        g_duel[iChallengerClient].iDuelArena = 5;
        g_duel[iChallengeeClient].iDuelArena = 5;
        g_bReserved5 = true;
    } else if (!g_bReserved6)
    {
        g_duel[iChallengerClient].iDuelArena = 6;
        g_duel[iChallengeeClient].iDuelArena = 6;
        g_bReserved6 = true;
    } else
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}No free duel arenas. Wait for one to be freed");
        return;
    }

    g_bHasChallenged[iChallengerClient] = true;
    g_bChallenged[iChallengeeClient] = true;

    g_duel[iChallengerClient].iDuelChallengerUserId = iChallengeeUserId;
    g_duel[iChallengeeClient].iDuelChallengerUserId = iChallengerUserId;

    char sChallengerName[64];
    GetClientName(iChallengerClient, sChallengerName, sizeof(sChallengerName));

    CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}Duel challenge sent to {lightgreen}%s", sChallengeeName);
    CPrintToChat(iChallengeeClient, "{yellow}[GFR] {lightgreen}%s {default}has sent you a duel challenge. {green}/accept {default}or {red}/decline", sChallengerName);

    // Play sound effect
    ClientCommand(iChallengeeClient, "playgamesound ui/duel_challenge.wav");

    // Timer
    g_hChallengeTimers[iChallengerClient] = CreateTimer(30.0, Timer_DuelChallenge, iChallengerUserId, TIMER_FLAG_NO_MAPCHANGE);
}

public Action Timer_DuelClock(Handle timer, int iChallengerUserId)
{
    int iChallengerClient = GetClientOfUserId(iChallengerUserId);
    int iChallengeeUserId = g_duel[iChallengerClient].iDuelChallengerUserId;
    int iChallengeeClient = GetClientOfUserId(iChallengeeUserId);

    g_duel[iChallengerClient].iDuelTimerSecs--;
    g_duel[iChallengeeClient].iDuelTimerSecs--;

    if (g_duel[iChallengerClient].iDuelTimerMins <= 0 && g_duel[iChallengerClient].iDuelTimerSecs <= 0)
    {
        // If duel's a tie when time ends next kill wins golden point
        if (g_duel[iChallengerClient].iDuelScore == g_duel[iChallengeeClient].iDuelScore)
        {
            g_duel[iChallengerClient].bDuelTimeOver = true;
            g_duel[iChallengeeClient].bDuelTimeOver = true;

            delete g_hDuelClockTimer[iChallengerClient];
            return Plugin_Stop;
        }

        // Show 0:00 time
        ShowScoreSpecDueler(iChallengerClient, iChallengeeClient);
        ShowScoreSpecDueler(iChallengeeClient, iChallengerClient);

        delete g_hChallengeTimers[iChallengeeClient];
        delete g_hDuelStartTimers[iChallengeeClient];
        delete g_hDuelScoreRefresh[iChallengeeClient];

        delete g_hChallengeTimers[iChallengerClient];
        delete g_hDuelStartTimers[iChallengerClient];
        delete g_hDuelScoreRefresh[iChallengerClient];

        g_bInDuel[iChallengeeClient] = false;
        g_bInDuel[iChallengerClient] = false;

        char sChallengerName[64];
        GetClientName(iChallengerClient, sChallengerName, sizeof(sChallengerName));
        char sChallengeeName[64];
        GetClientName(iChallengeeClient, sChallengeeName, sizeof(sChallengeeName));

        // Time runs out
        if (g_duel[iChallengerClient].iDuelScore > g_duel[iChallengeeClient].iDuelScore)
        {
            // Challenger wins

            DuelEnd(iChallengerClient, iChallengeeClient);

            CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}after time ran out. Score: {lightgreen}%d{default}-{lightgreen}%d", sChallengerName, sChallengeeName, g_duel[iChallengerClient].iDuelScore, g_duel[iChallengeeClient].iDuelScore);
        } else if (g_duel[iChallengeeClient].iDuelScore > g_duel[iChallengerClient].iDuelScore)
        {
            // Challengee wins

            DuelEnd(iChallengeeClient, iChallengerClient);

            CPrintToChatAll("{yellow}[GFR] {lightgreen}%s {default}has won a duel against {lightgreen}%s {default}after time ran out. Score: {lightgreen}%d{default}-{lightgreen}%d", sChallengeeName, sChallengerName, g_duel[iChallengeeClient].iDuelScore, g_duel[iChallengerClient].iDuelScore);
        }
        
        delete g_hDuelClockTimer[iChallengerClient];

        TF2_RespawnPlayer(iChallengerClient);
        TF2_RespawnPlayer(iChallengeeClient);
        
        return Plugin_Stop;
    }

    if (g_duel[iChallengerClient].iDuelTimerSecs < 0)
    {
        g_duel[iChallengerClient].iDuelTimerMins--;
        g_duel[iChallengerClient].iDuelTimerSecs = 59;

        g_duel[iChallengeeClient].iDuelTimerMins--;
        g_duel[iChallengeeClient].iDuelTimerSecs = 59;
    }

    return Plugin_Continue;
}

public Action Timer_DuelChallenge(Handle timer, int iChallengerUserId)
{
    int iChallengerClient = GetClientOfUserId(iChallengerUserId);
    if (!IsValidClient(iChallengerClient))
    {
        return Plugin_Handled;
    }

    int iChallengeeClient = GetClientOfUserId(g_duel[iChallengerClient].iDuelChallengerUserId);

    FreeArena(g_duel[iChallengerClient].iDuelArena);

    if (!IsValidClient(iChallengeeClient))
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}Player disconnected");
        return Plugin_Handled;
    }

    g_bHasChallenged[iChallengerClient] = false;
    g_bChallenged[iChallengeeClient] = false;

    g_duel[iChallengerClient].iDuelArena = 0;
    g_duel[iChallengeeClient].iDuelArena = 0;

    g_fDuelCooldown[iChallengerClient] = GetTickedTime();

    char sChallengerName[64];
    GetClientName(iChallengerClient, sChallengerName, sizeof(sChallengerName));

    char sChallengeeName[64];
    GetClientName(iChallengeeClient, sChallengeeName, sizeof(sChallengeeName));

    CPrintToChat(iChallengerClient, "{yellow}[GFR] {lightgreen}%s {default}didn't respond to the challenge in time", sChallengeeName);
    CPrintToChat(iChallengeeClient, "{yellow}[GFR] {default}You didn't respond to the duel sent by {lightgreen}%s {default}in time", sChallengerName);

    // Play sounds
    ClientCommand(iChallengerClient, "playgamesound ui/duel_challenge_rejected.wav");
    ClientCommand(iChallengeeClient, "playgamesound ui/duel_challenge_rejected.wav");

    delete g_hChallengeTimers[iChallengerClient];

    return Plugin_Handled;
}

void FreeArena(int arena)
{
    switch (arena)
    {
        case 1:
        {
            g_bReserved1 = false;
        }
        case 2:
        {
            g_bReserved2 = false;
        }
        case 3:
        {
            g_bReserved3 = false;
        }
        case 4:
        {
            g_bReserved4 = false;
        }
        case 5:
        {
            g_bReserved5 = false;
        }
        case 6:
        {
            g_bReserved6 = false;
        }
    }
}

public Action Timer_RefreshText(Handle timer, int iClientUserId)
{
    int client = GetClientOfUserId(iClientUserId);
    int challenger = GetClientOfUserId(g_duel[client].iDuelChallengerUserId);

    if (!IsValidClient(client))
    {
        return Plugin_Handled;
    }

    if (!IsValidClient(challenger))
    {
        return Plugin_Handled;
    }

    ShowScoreSpecDueler(client, challenger);

    return Plugin_Continue;
}

void ShowScoreSpecDueler(int iChallengerClient, int iChallengeeClient)
{
    ShowScore(iChallengerClient, iChallengerClient, iChallengeeClient);

    // Show score to spectator
    if (IsPlayerAlive(iChallengerClient))
    {
        for (int i = 1; i <= MaxClients; i++)
        {
            if (!IsValidClient(i))
            {
                continue;
            }

            if (!IsClientObserver(i))
            {
                continue;
            }

            // Spectator mode
            int iSpecMode = GetEntProp(i, Prop_Send, "m_iObserverMode");

            if (iSpecMode != SPECMODE_FIRSTPERSON && iSpecMode != SPECMODE_3RDPERSON)
            {
                continue;
            }

            // Spectator target
            int iTarget = GetEntPropEnt(i, Prop_Send, "m_hObserverTarget");

            if (iTarget == iChallengerClient)
            {
                ShowScore(i, iChallengerClient, iChallengeeClient);
            }
        }
    }
}

void ShowScore(int iClientShow, int iChallengerClient, int iChallengeeClient)
{
    int iSeconds = g_duel[iChallengerClient].iDuelTimerSecs;
    int iMinutes = g_duel[iChallengerClient].iDuelTimerMins;

    int iOwnScore = g_duel[iChallengerClient].iDuelScore;
    int iOwnKillstreak = g_duel[iChallengerClient].iDuelKillstreak;

    char sOwnName[64];
    sOwnName = g_duel[iChallengeeClient].sDuelChallengerName;

    int iChallengerScore = g_duel[iChallengeeClient].iDuelScore;
    int iChallengerKillstreak = g_duel[iChallengeeClient].iDuelKillstreak;

    char sChallengerName[64];
    sChallengerName = g_duel[iChallengerClient].sDuelChallengerName;

    if (g_duel[iChallengerClient].bDuelTimeOver)
    {
        // Golden point
        ShowSyncHudText(iClientShow, g_hHudSync, "Golden point\n%s: %d/10\nKS: %d/4\n\n%s: %d/10\nKS: %d/4", sOwnName, iOwnScore, iOwnKillstreak, sChallengerName, iChallengerScore, iChallengerKillstreak);
    } else
    {
        // 5:00
        // Name: 10/10
        // KS: 4/4
        // Name: 10/10
        // KS 4/4
        ShowSyncHudText(iClientShow, g_hHudSync, "%d:%02d\n%s: %d/10\nKS: %d/4\n\n%s: %d/10\nKS: %d/4", iMinutes, iSeconds, sOwnName, iOwnScore, iOwnKillstreak, sChallengerName, iChallengerScore, iChallengerKillstreak);
    }
}

void TeleportPlayerToArena(int iUserId)
{
    int client = GetClientOfUserId(iUserId);

    if (!IsValidClient(client))
    {
        return;
    }
    
    switch (g_duel[client].iDuelArena)
    {
        case 1:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin1, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin1, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
        case 2:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin2, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin2, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
        case 3:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin3, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin3, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
        case 4:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin4, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin4, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
        case 5:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin5, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin5, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
        case 6:
        {
            if (TF2_GetClientTeam(client) == TFTeam_Red)
            {
                TeleportEntity(client, g_fRedSpawnOrigin6, g_fRedSpawnAngle, NULL_VECTOR);
            } else
            {
                TeleportEntity(client, g_fBlueSpawnOrigin6, g_fBlueSpawnAngle, NULL_VECTOR);
            }
        }
    }
}

public Action Timer_DuelStart(Handle timer, int iChallengerUserId)
{
    int iChallengerClient = GetClientOfUserId(iChallengerUserId);

    int iChallengeeUserId = g_duel[iChallengerClient].iDuelChallengerUserId;
    int iChallengeeClient = GetClientOfUserId(iChallengeeUserId);

    if (!IsValidClient(iChallengerClient))
    {
        CPrintToChat(iChallengeeClient, "{yellow}[GFR] {default}Challenger disconnected");
        FreeArena(g_duel[iChallengeeClient].iDuelArena);
        return Plugin_Handled;
    }

    if (!IsValidClient(iChallengeeClient))
    {
        CPrintToChat(iChallengerClient, "{yellow}[GFR] {default}Challenger disconnected");
        FreeArena(g_duel[iChallengerClient].iDuelArena);
        return Plugin_Handled;
    }

    // Challenger
    g_duel[iChallengerClient].iDuelScore = 0;
    g_duel[iChallengerClient].iDuelKillstreak = 0;
    g_duel[iChallengerClient].iDuelTimerSecs = 0;
    g_duel[iChallengerClient].iDuelTimerMins = 5;

    char sChallengeeName[64];
    GetClientName(iChallengeeClient, sChallengeeName, 64);

    g_duel[iChallengerClient].sDuelChallengerName = sChallengeeName;

    char sChallengeeSteamId[64];
    GetClientAuthId(iChallengeeClient, AuthId_Steam2, sChallengeeSteamId, 64);

    g_duel[iChallengerClient].sDuelChallengerSteamId = sChallengeeSteamId;

    g_hDuelScoreRefresh[iChallengerClient] = CreateTimer(0.2, Timer_RefreshText, iChallengerUserId, TIMER_FLAG_NO_MAPCHANGE | TIMER_REPEAT);

    // Challengee
    g_duel[iChallengeeClient].iDuelScore = 0;
    g_duel[iChallengeeClient].iDuelKillstreak = 0;
    g_duel[iChallengeeClient].iDuelTimerSecs = 0;
    g_duel[iChallengeeClient].iDuelTimerMins = 5;

    char sChallengerName[64];
    GetClientName(iChallengerClient, sChallengerName, 64);

    g_duel[iChallengeeClient].sDuelChallengerName = sChallengerName;
    
    char sChallengerSteamId[64];
    GetClientAuthId(iChallengerClient, AuthId_Steam2, sChallengerSteamId, 64);

    g_duel[iChallengeeClient].sDuelChallengerSteamId = sChallengerSteamId;

    g_hDuelScoreRefresh[iChallengeeClient] = CreateTimer(0.2, Timer_RefreshText, iChallengeeUserId, TIMER_FLAG_NO_MAPCHANGE | TIMER_REPEAT);

    // Team check
    if (TF2_GetClientTeam(iChallengeeClient) == TFTeam_Red)
    {
        TF2_ChangeClientTeam(iChallengerClient, TFTeam_Blue);
        TF2_ChangeClientTeam(iChallengeeClient, TFTeam_Red);
    } else
    {
        TF2_ChangeClientTeam(iChallengerClient, TFTeam_Red);
        TF2_ChangeClientTeam(iChallengeeClient, TFTeam_Blue);
    }

    // Class check
    if (TF2_GetPlayerClass(iChallengerClient) != TFClass_Soldier)
    {
        TF2_SetPlayerClass(iChallengerClient, TFClass_Soldier, false, true);
    }

    if (TF2_GetPlayerClass(iChallengeeClient) != TFClass_Soldier)
    {
        TF2_SetPlayerClass(iChallengeeClient, TFClass_Soldier, false, true);
    }

    TF2_RespawnPlayer(iChallengerClient);
    TF2_RespawnPlayer(iChallengeeClient);

    g_bInDuel[iChallengerClient] = true;
    g_bInDuel[iChallengeeClient] = true;

    // Teleport players to duel arena
    TeleportPlayerToArena(iChallengerUserId);
    TeleportPlayerToArena(iChallengeeUserId);

    g_bHasChallenged[iChallengerClient] = false;
    g_bChallenged[iChallengeeClient] = false;

    delete g_hDuelStartTimers[iChallengerClient];

    g_hDuelClockTimer[iChallengerClient] = CreateTimer(1.0, Timer_DuelClock, iChallengerUserId, TIMER_FLAG_NO_MAPCHANGE | TIMER_REPEAT);

    CPrintToChatAll("{yellow}[GFR] {default}A duel between {lightgreen}%s {default}and {lightgreen}%s {default}has started", sChallengerName, sChallengeeName);

    return Plugin_Handled;
}

public Action CmdListener_Suicide(int client, const char[] command, int argc)
{
    if (g_bInDuel[client])
    {
        CPrintToChat(client, "{yellow}[GFR] {default}Suicide disabled in duel");
        return Plugin_Handled;
    }
    return Plugin_Continue;
}

bool IsValidClient(int client)
{
    if (0 < client && client <= MaxClients)
    {
        if (IsClientInGame(client))
        {
            return true;
        }
    }
    return false; 
}