#include <string.h>
#include "conf.h"
#include "../ui/ui.h"
#include "vec3.h"
#include "vec2.h"

//!any given file is expected to be opened.

void get_anchor(enum anchor *, char *);


void get_anchor(enum anchor *a, char *s)
{
	if (strcmp(s, "top_left") == 0)
		*a = TOP_LEFT;
	else if (strcmp(s, "top_middle") == 0)
		*a = TOP_MIDDLE;
	else if (strcmp(s, "top_right") == 0)
		*a = TOP_RIGHT;
	else if (strcmp(s, "middle_left") == 0)
		*a = MIDDLE_LEFT;
	else if (strcmp(s, "middle_middle") == 0)
		*a = MIDDLE_MIDDLE;
	else if (strcmp(s, "middle_right") == 0)
		*a = MIDDLE_RIGHT;
	else if (strcmp(s, "bottom_left") == 0)
		*a = BOTTOM_LEFT;
	else if (strcmp(s, "bottom_middle") == 0)
		*a = BOTTOM_MIDDLE;
	else if (strcmp(s, "bottom_right") == 0)
		*a = BOTTOM_RIGHT;
}

/*
 * read the next "block" from a given config file.
 * return 1 if the end of
 * file or a marker has been reached, else 0.
 * 
 * *s is expected to be combinations of key=type
 * separated by spaces.
 * ex: "position=3 scale=3 self_anchor=a"
 * 
 * the function will then fill any other parameters 
 * given (expected to be pointers of different types)
 * with the value associated with the key, if it's
 * been found in the config block.
 * otherwise, the parameter stays UNTOUCHED.
 * 
 * You MUST NOT directly feed this function a string ("blah")
 * as it needs to be able to modify it. directly giving it a
 * string makes this string read-only. Pass a pointer to an
 * array instead.
 * 
 * For now, the function doesn't check if you've passed
 * the correct number of arguments related to the string.
 * Please use this function with care.
 */
int conf_get(FILE *f, char *s, ...)
{
	va_list ap;
	int marker = 0;
	char *eof;
	char line[128];


	//get a line from the config block
	while ((eof = fgets(line, sizeof(line), f)) != NULL) {

		if (line[0] == '#' || line[0] == '\n')
			continue;
		if (line[0] == '}')
			break;
		//if we reach a marker, return 1 (it means we got to the end of the ui section)
		if (line[0] == '@') {
			marker = 1;
			break;
		}

		//get the keypair from the line
		char f_key[64], value[64];
		if (sscanf(line, "%[^=]=%[^\n]", f_key, value) != 2) {
			fprintf(stderr, "Error: could not find a proper keypair in %s\n", line);
			continue;
		}

		va_start(ap, s);
		int found = 0;

		//since the function strtok modifies the source string, copy the passed string
		//to this variable as we need to use it multiple times
		char cs[128];
		memcpy(cs, s, sizeof(cs));

		//it seems that by default, ap points to the first argument (the passed string) Nothing needs
		//to be done here, so incrementing it here makes it so that our pointer points to the
		//first variadic parameter.
		void *ptr = va_arg(ap, char *);


		//compare the key name to each word in the passed string
		//strtok: replaces the first delimiter found by \0, and returns a pointer to the first
		//word before it.
		char *w = strtok(cs, " ");
		//strtok will return NULL when at the end of the string.
		while (w != NULL) {
			char s_key[64], type;
			//scan the word that strtok found, and retrieve it's key and type
			if (sscanf(w, "%[^=]=%c", s_key, &type) != 2) {
				fprintf(stderr, "Error: could not find proper values in the passed string\n");
				continue;
			}

			//printf("w=%s\n", w);

			//if the current key from the passed string matches the current one from the file...
			if (strcmp(s_key, f_key) == 0) {
				found = 1;
			}

			switch (type) {
			case 'a':
				if (found) {
					enum anchor *a = (enum anchor *)ptr;
					//*a = MIDDLE_MIDDLE;
					get_anchor(a, value);
				}
				ptr = va_arg(ap, enum anchor *);
				break;
			case '3':
				if (found) {
					struct vec3 *v3 = (struct vec3 *)ptr;
					vec3_from_string(v3, value);
				}
				ptr = va_arg(ap, struct vec3 *);
				break;
			case '2':
				if (found) {
					struct vec2 *v2 = (struct vec2 *)ptr;
					vec2_from_string(v2, value);
				}
				ptr = va_arg(ap, struct vec2 *);
				break;
			case 's':
				if (found) {
					memcpy((char*)ptr, value, sizeof(char) * 64);
				}
				ptr = va_arg(ap, char *);
				break;
			default:
				fprintf(stderr, "Error: icrementing argument pointer by int*,\
				format \"%c\" not known\n", type);
				ptr = va_arg(ap, int *);
				break;
			}

			//if the key has been found, then we dont need to do anything more on this line
			//and so we can break.
			if (found)
				break;
			//if the key hasn't been found however, redo the process with a new key from the passed
			//string.
			w = strtok(NULL, " ");
		}

		va_end(ap);
	}
	if (marker || eof == NULL)
		return 1;
	return 0;
}

//return the type of the next block. return T_UNKOWN when
//encountered a marker or an unkown line.
enum block_type conf_get_type(FILE *f)
{
	char *eof;
	char line[128];
	char t[64];

	while ((eof = fgets(line, sizeof(line), f)) != NULL) {

		if (line[0] == '#' || line[0] == '\n')
			continue;
		else if (line[0] == '@') {
			break;
		}
		if (sscanf(line, "%s {", t) == 1) {
			if (strcmp(t, "skybox") == 0)
				return T_SKYBOX;
			else if (strcmp(t, "button") == 0)
				return T_UI_BUTTON;
		}

		fprintf(stderr, "Error: unkown line encountered: %s\n", line);
		break;
	}
	return T_UNKOWN;
}


//set the file position indicator to the line
//below the marker (ex: @main)
//return 0 if successful, 1 if the marker couldn't
//be found.
int conf_goto_marker(FILE *file, char *marker)
{
	rewind(file);
	char m[64];
	char line[128];
	while (fgets(line, sizeof(line), file) != NULL) {
		sscanf(line, "%s\n", m);
		if (strcmp(m, marker) == 0)
			return 0;
	}
	return 1;
}


