#include <stdlib.h>

#include "text.h"
#include "../loader/texture.h"
#include "../loader/vao.h"
#include "../loader/vbo.h"
#include "../utils/vec3.h"
#include "../utils/vec2.h"
#include "../utils/csv.h"
#include "../utils/cfg.h"

//default values
#define TEXT_DEF_FONT "FONT_DEF"
#define TEXT_DEF_TEX "TEX_DEF_TEXT" //haha

//all the letters will be stored here along their vbo index
static struct letter_info letters[TEXT_MAX_LETTERS];
//the list of strings whose pointer is given to the renderer
static struct string_info strings[TEXT_MAX_STRINGS] = {0};
static char filepath[128] = {0};

//for now, the code only supports ONE font.
static unsigned int vbo_handle; //default vbo
static unsigned int vao_handle; //default vao
static struct texture_info *texture; //default texture
static struct shp_info *shprogram; //default shprogram for all text

static float vert_buf[6912]; //96 * 9 * 8, hard-coded and ugly. the buffer that will be given to the VBO.
static int float_count = 0; //to keep track of the number of floats in the buffer above

//load the default (and for now, only) font into the program.
int text_initialize(void)
{
	int vbo_index = 0;

	vao_create(&vao_handle);
	vao_bind(vao_handle);


	//---load the vertices (model + UVs)---//

	char * csv_string = csv_parse("assets/fonts_index.csv", "font_name", TEXT_DEF_FONT);
	if (csv_string == NULL)
	{
		fprintf(stderr, "Error: could not find the font \"%s\"", TEXT_DEF_FONT);
		return -1;
	}
	sscanf(csv_string, "%*[^;];%*u;%[^;];%*[^\n]", filepath);
	FILE *f = fopen(filepath, "r");

	char key[CFG_MAX_KEY_LENGTH] = {0};
	char val[CFG_MAX_VALUE_LENGTH] = {0};
	int rv;
	while ((rv = cfg_read(f, key, val)) != EOF) {
		if (rv == 1) { continue; }
		//printf("KEY: \"%s\" VALUE: \"%s\"\n", key, val);

		//the index of the letter is the letter's decimal number. ASCII only!
		struct letter_info *l = &letters[(int)key[0]];

		//as we know cfg files for fonts only have 1 char for the key, we take the first byte of the key
		//of course, this only works with ASCII characters! UTF8 has tons of chars that are 2 or 3 bytes.
		l->letter = key[0];

		//a letter is composed of 6 elements. (we can say 6 vertices, but it's a bit different.)
		l->vbo_index = vbo_index;
		vbo_index += 6; //increment for the next letter

		//get the uv coordinates out of the value.
		struct vec2 uv1, uv2, uv3, uv4;
		rv = sscanf(val, "%f %f %f %f %f %f %f %f", &uv1.x, &uv1.y, &uv2.x, &uv2.y, &uv3.x, &uv3.y, &uv4.x, &uv4.y);

		//vertices for the letter model, 2 triangles = square
		struct vec3 p1 = {0.0025, -0.005, 0},
			    p2 = {-0.0025, 0.005, 0},
			    p3 = {-0.0025, -0.005, 0},//face 1
			    p4 = {0.0025, -0.005, 0},
			    p5 = {0.0025, 0.005, 0},
			    p6 = {-0.0025, 0.005, 0};//face 2

/*
 * This below is to show which point needs to be paired with which UV. As you can see, the
 * 2 triangles form a square, on which the letter at the given UV coordinates will be displayed.
 *
 *         ^ +y
 *    -x < + > +x
 *         v -y
 *
 *     uv1        uv2
 *   p2   p6------p5
 *   |\    \      |
 *   | \    \     |
 *   |  \    \    |
 *   |   \-><-\   |
 *   |    \    \  |
 *   |     \    \ |
 *   |      \    \|
 *   p3------p1   p4
 *   uv3       uv4
 */

		//put into the vert_buf all the vertices with their corresponding UVs
		vert_buf[float_count++] = p1.x; vert_buf[float_count++] = p1.y; vert_buf[float_count++] = p1.z;
		vert_buf[float_count++] = uv4.x; vert_buf[float_count++] = uv4.y;

		vert_buf[float_count++] = p2.x; vert_buf[float_count++] = p2.y; vert_buf[float_count++] = p2.z;
		vert_buf[float_count++] = uv1.x; vert_buf[float_count++] = uv1.y;

		vert_buf[float_count++] = p3.x; vert_buf[float_count++] = p3.y; vert_buf[float_count++] = p3.z;
		vert_buf[float_count++] = uv3.x; vert_buf[float_count++] = uv3.y;

		vert_buf[float_count++] = p4.x; vert_buf[float_count++] = p4.y; vert_buf[float_count++] = p4.z;
		vert_buf[float_count++] = uv4.x; vert_buf[float_count++] = uv4.y;

		vert_buf[float_count++] = p5.x; vert_buf[float_count++] = p5.y; vert_buf[float_count++] = p5.z;
		vert_buf[float_count++] = uv2.x; vert_buf[float_count++] = uv2.y;

		vert_buf[float_count++] = p6.x; vert_buf[float_count++] = p6.y; vert_buf[float_count++] = p6.z;
		vert_buf[float_count++] = uv1.x; vert_buf[float_count++] = uv1.y;

	}
	fclose(f);

	vbo_create(&vbo_handle);
	vbo_fill_new(vbo_handle, vert_buf, float_count, GL_STATIC_DRAW, VBO_TEXT);

	//load the texture
	texture_allocate(&texture);
	texture_search(TEXT_DEF_TEX, texture);
	texture_load(texture);

	texture_bind(texture);
	texture_smooth(0);
	texture_unbind();

	//load the shprogram
	shprogram_search("SHP_TEXT", &shprogram);
	shprogram_set_uniforms(shprogram);

	vao_unbind();
	return 0;
}

//create a new string, at the screen position pos, of color col, of size size,
//containing the chars present in text.
//returns the index of the created string if no errors occured, -1 otherwise.
int text_new(struct vec3 pos, struct vec3 rgb, struct vec3 size, char *text)
{
	int i;
	for (i = 0; i < TEXT_MAX_STRINGS; i++) {
		if (strings[i].used == 0) {
			struct string_info *s = &strings[i];
			s->used = 1;
			s->color = vec3_rgb_clamp(rgb);
			s->position = pos;
			s->letter_count = 0;
			s->vao_handle = vao_handle;
			s->vbo_handle = vbo_handle;
			s->shprogram = shprogram;
			s->texture = texture;
			s->model_matrix = mat4_identity(); //isn't it ressource intensive to do it here?
			mat4_scale(s->model_matrix, size);
			mat4_position(s->model_matrix, pos);

			s->letter_width = 0.005 * size.x; //0.010 = the width of the hard-coded model in X axis

			for (int e = 0; e < TEXT_MAX_LETTERS; e++) {
				if (text[e] == '\0')
					return i;
				s->letter_count++;
				s->letters[e] = letters[(int)text[e]].vbo_index; //only ASCII
			}
		}
	}
	return -1;
}

struct string_info * text_get_pointer(void)
{
	return &strings[0];
}

//stop rendering a string and make it's slot available again.
//takes handle (index) to an existing string.
void text_delete(int handle)
{
	if (handle == -1) //if the string is invalid to begin with, don't do anything.
		return;
	
	strings[handle].used = 0;
}

//update the text inside a string designated by its handle.
void text_update(int handle, char *text)
{
	if (handle == -1) //if the string is invalid to begin with, don't do anything.
		return;
	struct string_info *t = &strings[handle];
	t->letter_count = 0;
	for (int e = 0; e < TEXT_MAX_LETTERS; e++) {
		if (text[e] == '\0') { return; }
		t->letter_count++;
		t->letters[e] = letters[(int)text[e]].vbo_index;
	}
}

void text_append_char(int handle, char c)
{
	if (handle == -1)
		return;
	struct string_info *t = &strings[handle];
	if (t->letter_count >= TEXT_MAX_LETTERS) //no more room for another char
		return;
	t->letters[t->letter_count++] = letters[(int)c].vbo_index;
}

void text_remove_chars(int handle, int count)
{
	if (handle == -1)
		return;
	struct string_info *t = &strings[handle];
	t->letter_count -= count;
	if (t->letter_count < 0)
		t->letter_count = 0;
}

void text_empty(int handle)
{
	if (handle == -1)
		return;
	strings[handle].letter_count = 0;
}

void text_reset_pos(int handle, struct vec3 pos)
{
	if (handle == -1) { return; }
	strings[handle].position = pos;
	mat4_position(strings[handle].model_matrix, pos);
}

void text_move(int handle, struct vec3 pos)
{
	if (handle == -1) { return; }
	strings[handle].position = vec3_add(strings[handle].position, pos);
	mat4_translate(strings[handle].model_matrix, pos);
}

void text_color(int handle, struct vec3 rgb)
{
	if (handle == -1) { return; }
	strings[handle].color = vec3_rgb_clamp(rgb);
}
