/*
 * prot.* takes care of filling up buffers with data and decoding arriving ones.
 * The code of these files is most likely not portable to other platforms as it
 * assumes ints and floats to be 4 bytes long. Moreover, it does not (yet) convert
 * data from little endian to big (network) endian.
 */

#ifndef PROT_H
#define PROT_H

#include "../utils/vec3.h"
#include "../utils/quat.h"
#include "nconf.h"

/* 
 * Payloads of data received are composed of what I call "bundles" of data,
 * each containing information sent by one client. A typical payload could
 * look like this:
 *
 * D_START | pid | data type | data | D_END | D_START | pid | data type | ...
 *
 * "pid" is the Player ID, which tells who sent the bundle. Each client receives
 * one when they connect to a server.
 * "data type" is a number describing the data that will follow.
 */

#define D_START		0xF000	//start of a bundle of data
#define D_END		0xFF00	//end of a bundle
#define D_SRV		0xFFF0	//pid for the server
#define D_PLAYER	0x000A	//indicates the data coming is player data
#define D_PID		0x000B	//the next data is the client's pid
#define D_NEWPLAYER	0x000C	//a new player joined the game
#define D_PLAYERQUIT    0x000D	//the following player left the game
#define D_NEWPROJECTILE 0x000E  //a projectile has been fired
#define D_NEWMESSAGE    0x000F  //a new message needs to be put into the chat

struct ndata {
	char buf[PROT_BYTE_LENGTH];
	int size;
};

//data of decoded bundles are stored in this struct.
struct dtypes {
	int data_n;						//the number of payloads received
	int data_t[PROT_MAX_BUNDLES];				//the type of data received
	int sender[PROT_MAX_BUNDLES];				//the (pid of the) sender of the data 
	int pid[PROT_MAX_BUNDLES][MAX_PLAYERS];			//the pid(s) the data is about
	int team[PROT_MAX_BUNDLES];				//the team of the player (used with D_NEWPLAYER)
	struct vec3 position[PROT_MAX_BUNDLES][MAX_PLAYERS];	//positions
	struct vec3 velocity[PROT_MAX_BUNDLES][MAX_PLAYERS];	//velocities
	float speed[PROT_MAX_BUNDLES];				//speed
	struct quat q[PROT_MAX_BUNDLES][MAX_PLAYERS];		//rotations
	char message[PROT_MAX_BUNDLES];				//sent messages
};


int prot_head(struct ndata *, int);
int prot_fill(struct ndata *, int, char *, int);
void prot_clear(struct ndata *);

int prot_decode(struct ndata *, struct dtypes *);
int prot_clt_info(struct ndata *, int, int); //temporary function - to change!
void prot_quit_notice(struct ndata *, int);



#endif
