/* This is the file containing all the online players code. This file is shared between
 * the client and the server.
 */

#include <stdio.h>
#include <stdlib.h>

#include "oplayers.h"
#include "../utils/vec3.h"
#include "../loader/loader.h"
#include "../lists/proplist.h"
#include "clt.h"
#include "nconf.h"

enum player_team {
	TEAM_RED,
	TEAM_BLUE,
	TEAM_GREEN
};

static int player_team_color[] =  {0xFC0036, 0x63e0f9, 0xA6FC44}; //team colors

static int player_team_count[MAX_TEAMS] = {0};

//the struct array that will contain player info
static struct player_info *players;
static int player_count = 0;

static int get_lowest_team_count(void);


//calloc the necessary memory for the players and initialise some
//variables.
int oplayers_init(void)
{
	players = calloc(MAX_PLAYERS, sizeof(struct player_info));
	
	//the server's player, as in srv.c/srv_initialize(), is on team 0 by default. increment the team count
	//here so that the teams are properly balanced.
	player_team_count[0] = 1;
	player_count = 1; //server's player is already here

	return 0;
}

//add a player. This function is used by the server, and determines if there is
//enough room for a new player.
struct player_info * oplayers_add(int socket)
{
	//the slot 0 is reserved to the server's player. nothing is actually in it, but the
	//server's player identifies itself to other clients as PID 0.
	for (int i = 1; i < MAX_PLAYERS; i++) {
		if (players[i].socket == 0) {
			int team = get_lowest_team_count();
			oplayers_add_player(i, socket, team);
			return &players[i];
		}
	}
	return NULL;
}

//return the player associated with the given pid.
struct player_info * oplayers_get_player_info(int pid)
{
	if (players[pid].socket != 0)
		return &players[pid];
	return NULL;
}

//remove a player from the index.
int oplayers_remove(int pid)
{
	if (players[pid].socket == 0)
		return -1;

	printf("player %d quit\n", pid);
	loader_unload(players[pid].prop);
	players[pid].socket = 0;
	players[pid].prop = NULL;
	player_count--;
	player_team_count[players[pid].team]--;
	players[pid].team = 0;
	return 0;
}

//add a player in the player index in the given slot. should not be called directly by the server.
int oplayers_add_player(int slot, int socket, int team)
{
	//if we want to add ourselves, stop immediately
	if (slot == clt_get_pid())
		return -1;

	printf("NEW PLAYER %d\n", slot);
	int id;
	if ((id = loader_load_asset_new("ENT_OPLAYER", vec3(0, 0, 0), T_OPLAYER)) < 0) {
		fprintf(stderr, "Error: could not generate the player model\n");
		return -1;
	}
	struct prop_info *p;
	p = proplist_get_by_index(id);
	//printf("ID OF PLAYER PROP: %d\n", id);

	players[slot].prop = p;
	players[slot].socket = socket;
	players[slot].team = team;
	players[slot].pid = slot; //the pid is at the same time the slot
	player_count++;
	player_team_count[team]++;
	p->color = vec3_hex_to_rgb(player_team_color[team]);
	//prop_random_color(p);
	printf("SOCKET = %d\n", socket);

	return 0;
}

//return the socket associated with the given pid.
int oplayers_get_pid_from_socket(int socket)
{
	int pid;
	for (pid = 0; pid < MAX_PLAYERS; pid++) {
		if (players[pid].socket == socket)
		return pid;
	}
	return -1;
}

int oplayers_get_socket_from_pid(int pid)
{
	return players[pid].socket;
}

struct prop_info * oplayers_get_prop_from_pid(int pid)
{
	return players[pid].prop;
}

int oplayers_get_team_from_pid(int pid)
{
	return players[pid].team;
}

//update the player (whose pid is referenced in dtypes->sender).
void oplayers_update(struct dtypes *dt, int index)
{
	if (players[dt->sender[index]].pid == clt_get_pid())
		return;
	if (players[dt->sender[index]].prop == NULL) {
		//if no prop is associated with the pid, it likely means that no player is at this pid. This
		//should not happen, so send an error message and abandon.
		printf("prop of %d not existing\n", index);
		return;
	}
	prop_reset_pos(players[dt->sender[index]].prop, dt->position[index][0]);
	prop_rotate(players[dt->sender[index]].prop, dt->q[index][0]);
}

//return the enum of the team with the fewest players
static int get_lowest_team_count(void)
{
	int players = 999, team_count = TEAM_RED; //default team to be returned

	for (int i = 0; i < MAX_TEAMS; i++) {
		if (player_team_count[i] < players) {
			team_count = i;
			players = player_team_count[i];
		}
	}
	return team_count;
}

void oplayers_clear(void)
{
	for (int i = 0; i < MAX_PLAYERS; i++) {
		oplayers_remove(i);
	}
}
