#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "map2.h"
#include "../loader/loader.h"
#include "../lists/proplist.h"
#include "../utils/csv.h"
#include "../utils/str.h"
#include "../utils/states.h"
#include "../utils/io.h"

//various structures to store temporary data
static struct vec3 pos = {0,0,0};
static struct vec3 rot = {0,0,0}; //should be converted to a quaternion later
static char map_path[MAP2_MAX_PATH_LENGTH];

enum map2t {
	MAP2_PROP,
	MAP2_SKYBOX,
	MAP2_PLAYER,
	MAP2_UNKOWN,
	MAP2_UNIMPLEMENTED,
};


static int read(FILE *);
static void mv_to_char(FILE *, char);
static int get_infos(char *, char *);
static int get_params(char *);


int map2_load(char *map_name)
{
	//the csv utilities are dying to be rewritten...
	char *csv_string = csv_parse("assets/maps_index.csv", "name", map_name);
	if (csv_string == NULL) {
		prnte("Error: could not find info on the map %s\n", map_name);
		return -1;
	}
	sscanf(csv_string, "%*[^;];%*d;%[^\n]\n", map_path);

	printf("---loading %s---\n", map_name);

	FILE *stream;
	if ((stream = fopen(map_path, "rb")) == NULL) {
		prnte("Error: could not open the map file\n");
		return -1;
	}

	if (read(stream) < 0) {
		prnte("Error: occured when loading the map\n");
		fclose(stream);
		return -1;
	}
	fclose(stream);
	return 0;
}

//read the entire file, filter its content and
//send it to the map_interpret() function.
static int read(FILE *f)
{
	unsigned int i = 0, line = 0, char_count = 0, char_count_line = 0;
	char buf[MAP2_MAX_ITEM_LENGTH];
	char c, lc; //current char, last char

	while ((c = fgetc(f)) != EOF) {
		switch (c) {
		case '\r':
			//fall through
		case ',':
			break;
		case '/':
			if (lc == '/') //comment
				mv_to_char(f, '\n');
			break;
		case '*':
			if (lc == '/') /*inline comment*/
				mv_to_char(f, '/');
			break;
		case ';':
			buf[i] = '\0';
			map_interpret(buf, line, char_count_line);
			i = 0;
			break;

		case '\n':
			line++;
			char_count_line = 0;
			//fall through
		case '\t':
			c = ' ';
			//fall through
		case ' ':
			if (lc == ' ')
				break;
			//fall through
		default:
			//all good
			buf[i++] = c;
			break;
		}
		lc = c;
		char_count++;
		char_count_line++;
	}
	printf("finished reading map: %d lines, %d characters\n", line, char_count);
	return 0;
}


//move the file indicator to the next instance of the given character,
static void mv_to_char(FILE *f, char p)
{
	char c;
	while ((c = fgetc(f)) != EOF)
		if (c == p) { return; }
}


void map_interpret(char *buf, unsigned int line, unsigned int char_count_line)
{
	//DO NOT FORGET TO MALLOC THE NAME OF THE PROP! Or the name will be shared with all the props!
	//this should be changed. the loader should not expect a passed variable to be available forever.
	char *item_name = malloc(MAP2_MAX_WORD_LENGTH * sizeof(char));

	int i = 0;
	while (buf[i] == ' ') { i++; }
	buf += i;

	switch (get_infos(buf, item_name)) {
	case MAP2_PROP:
		loader_load_asset_new(item_name, pos, T_MAP_PROP);
		break;
	case MAP2_SKYBOX:
		loader_load_asset_new(item_name, pos, SKYBOX);
		break;
	case MAP2_PLAYER:
		loader_load_asset_new(item_name, pos, PLAYER);
		break;
	case MAP2_UNIMPLEMENTED:
		break;
	default:
		prnte("Error: could not parse data around line %d: %s\n", line, buf);
		//load an error prop?
		free(item_name); //for now...
		break;
	}
}

//fill the global static variables with data parsed in the given string.
//return the type of the data.
static int get_infos(char *buf, char *item_name)
{
	int t = MAP2_UNKOWN;
	char type[MAP2_MAX_WORD_LENGTH];

	//get the type (eg: prop, skybox...) and the name (eg: PROP_ROCK...)
	int index = 0;
	index += str_get_word(type, buf);
	index += str_get_word(item_name, buf+index+1); //+1 because of the space after the 1st word
	buf += index+1; //move the pointer past the type and name


	//-----type parsing-----//

	if (strcmp(type, "prop") == 0) {
		t = MAP2_PROP;
	} else if (strcmp(type, "skybox") == 0) {
		t = MAP2_SKYBOX;
	} else if (strcmp(type, "player") == 0) {
		t = MAP2_PLAYER;
	} else if (strcmp(type, "light") == 0) {
		return MAP2_UNIMPLEMENTED; //lights are not implemented yet
	}
	
	if (get_params(buf) < 0)
		return MAP2_UNKOWN;
 	return t;
}

//recursive function which gets all the parameters in a given string.
//return 0 if it reached the end of the available parameters.
//return -1 if an error occured and it had to abort.
static int get_params(char *buf)
{
	int i = 0;
	while (buf[i] == ' ') { i++; } //sometimes there can be 2 spaces.
	buf += i;

	char type[MAP2_MAX_WORD_LENGTH];
	int index;

	//get the next type (eg: pos, rot...)
	if ((index = str_get_word(type, buf)) == 0 )//no more parameters to go through
		return 0;
	buf += index+1; //move the pointer after the word (+1 counting the space)


	//-----parameter parsing-----//
	
	if (strcmp(type, "pos") == 0) {//vec3 position


		if (sscanf(buf, "%f %f %f", &pos.x, &pos.y, &pos.z) != 3)
			return -1;
		//move the pointer past the 3 floats
		if (buf += str_get_char_index(buf, ' ', 2) < 0) 
			return 0; //end of string

	} else if (strcmp(type, "rot") == 0) {//vec3 rotation (must be converted into a quaternion)


		if (sscanf(buf, "%f %f %f", &rot.x, &rot.y, &rot.z) != 3)
			return -1;
		if (buf += str_get_char_index(buf, ' ', 2) < 0)
			return 0;


	} else if (strcmp(type, "col") == 0) {//vec3 color
		return 0; //colors are not implemented yet
	} else if (strcmp(type, "scale") == 0) {//vec3 scale
		return 0; //scales are not implemented yet
	} else if (strcmp(type, "col") == 0) {//int id
		return 0; //ids are not implemented yet
	} else if (strcmp(type, "script") == 0) {//script
		return 0; //scripts are not implemented yet
	} else { //unkown parameter!
		return -1;
	}

	//if one of the recursive calls returned -1, this one will as well
	if (get_params(buf) < 0)
		return -1;
	return 0;
}

char *map_get_path(void)
{
	return map_path;
}
