#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "texture.h"
#include "../utils/csv.h"

static GLuint load_dds(char *file_path);

void texture_allocate(struct texture_info **tex_pointer)
{
	*tex_pointer = malloc(sizeof(struct texture_info));
	struct texture_info *current_texture = *tex_pointer;
	current_texture->name = malloc(sizeof(char) * 128);
	current_texture->file_path = malloc(sizeof(char) * 128);
	current_texture->type = malloc(sizeof(char) * 128);
	current_texture->handle = 0;
}

int texture_search(char *name, struct texture_info *current_texture)
{
	printf("searching for information on texture %s...\n", name);

	char *csv_string = csv_parse("assets/textures_index.csv", "texture_name", name);
	if (csv_string == NULL) {
		fprintf(stderr, "Error: could not find the texture in the index\n");
		return -1;
	}

	sscanf(csv_string, "%[^;];%u;%[^;];%[^\n]",
	current_texture->name,
	&current_texture->id,
	current_texture->file_path,
	current_texture->type);

	return 0;
}

int texture_load(struct texture_info *current_texture)
{
	if (strcmp(current_texture->type, "dds") == 0) {
		current_texture->handle = load_dds(current_texture->file_path);
	} else {
		fprintf(stderr, "Error: the texture type in unkown\n");
		return -1;
	}

	if (current_texture->handle == 0) {
		return -1;
	}
	printf("%d\n", current_texture->handle);
	return 0;
}

//in ascii, means DXT1, DXT3 and DXT5, respectively
#define FOURCC_DXT1 0x31545844
#define FOURCC_DXT3 0x33545844
#define FOURCC_DXT5 0x35545844

//https://www.opengl-tutorial.org/beginners-tutorials/tutorial-5-a-textured-cube/
static GLuint load_dds(char *file_path)
{
	char header[128]; //i added 4
	FILE *tex_file;

	if ((tex_file = fopen(file_path, "rb")) == NULL) {
		fprintf(stderr, "Error: could not open the texture file\n");
		return 0;
	}

	//read one string of data of 128 bytes from the stream tex_file
	fread(&header, 128, 1, tex_file); //i added 4 to 124

	//declaration of the variables that will hold the informations contained
	//in the texture header
	//
	//Wait, every value is -4 what it should actually be! Changed it, but wth
	unsigned int height = *(unsigned int *)&(header[12]);
	unsigned int width = *(unsigned int *)&(header[16]);
	unsigned int linear_size = *(unsigned int *)&(header[20]);
	unsigned int mipmap_count = *(unsigned int *)&(header[28]);
	//It apparently wasn't 80 but 84. Open up a .dds file and look where is is written "DXT#"
	unsigned int fourcc = *(unsigned int *)&(header[84]);


	//read the texture file
	unsigned int buffer_size = mipmap_count > 1 ? linear_size * 2 : linear_size;
	char *buffer = malloc(buffer_size * sizeof(char));

	fread(buffer, 1, buffer_size, tex_file);
	fclose(tex_file);


	//determines the compression algoritm used on the texture
	//unsigned int components = (fourcc == FOURCC_DXT1) ? 3 : 4;
	unsigned int format;
	switch(fourcc) {
	case FOURCC_DXT1:
		format = GL_COMPRESSED_RGBA_S3TC_DXT1_EXT;
		break;
	case FOURCC_DXT3:
		format = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT;
		break;
	case FOURCC_DXT5:
		format = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;
		break;
	default:
		fprintf(stderr, "Error: could not understand the compression of the texture\n");
		free(buffer);
		return 0;
	}


	//create the texture buffer
	GLuint texture;
	glGenTextures(1, &texture);
	glPixelStorei(GL_UNPACK_ALIGNMENT,1);

	//for now, models only use one texture. This texture will be at the texture unit 0, if
	//I undertand correctly
	glActiveTexture(GL_TEXTURE0);

	//bind the current texture
	glBindTexture(GL_TEXTURE_2D, texture);


	unsigned int blocksize = (format == GL_COMPRESSED_RGBA_S3TC_DXT1_EXT) ? 8 : 16;
	unsigned int offset = 0;

	//load the mipmaps included with the texture
	for (unsigned int level = 0; level < mipmap_count && (width || height); level++) {
		unsigned int size = ((width + 3) / 4) * ((height + 3) / 4) * blocksize;

		glCompressedTexImage2D(GL_TEXTURE_2D, level, format, width, height, 0,
			size, buffer + offset);

		offset +=size;
		width /=2;
		height /=2;
	}
	free(buffer);

	glBindTexture(GL_TEXTURE_2D, 0);

	return texture;
}

void texture_bind(struct texture_info *current_texture)
{
	glBindTexture(GL_TEXTURE_2D, current_texture->handle);
}

void texture_unbind(void)
{
	glBindTexture(GL_TEXTURE_2D, 0);
}

void texture_deallocate(struct texture_info *current_texture)
{
	free(current_texture->name);
	free(current_texture->file_path);
	free(current_texture->type);
	if (current_texture->handle != 0) {
		//delete the texture buffer here
	}
	free(current_texture);
}

//enable/disable texture smoothing (magnification) on the currently bound texture.
void texture_smooth(int y)
{
	if (y)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	else
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
}
