#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <GL/glew.h>
#include "shprogram.h"
#include "shader.h"
#include "../game/camera.h"
#include "../utils/vec4.h"
#include "../utils/vec3.h"
#include "../utils/mat4.h"
#include "../utils/csv.h"
#include "../lists/proplist.h"

static struct prop_info **props_index;

static float *model_matrix;
static float *view_matrix;
static float *projection_matrix;
static float *mv_matrix;
static float *mvp_matrix;

GLint uniform_projection_matrix;
GLint uniform_light_position;

void shprogram_initialize(void)
{
	printf("Initializing matrices...\n");
	model_matrix = mat4_identity();
	mv_matrix = mat4_identity();
	mvp_matrix = mat4_identity();

	//get the projection and the view matrix from the camera
	projection_matrix = camera_get_projection_matrix_pointer();
	view_matrix = camera_get_view_matrix_pointer();


	//changing the order doesn't do a thing. why?
	mat4_multiply(view_matrix, model_matrix, mv_matrix);
	//oh, it matters here.
	//mat4_multiply(projection_matrix, mv_matrix, mvp_matrix);
	

}

int shprogram_allocate(struct shp_info **shp)
{
	*shp = malloc(sizeof(struct shp_info));
	(*shp)->name = malloc(sizeof(char) * 128);
	(*shp)->vert_file_path = malloc(sizeof(char) * 128);
	(*shp)->frag_file_path = malloc(sizeof(char) * 128);
	(*shp)->used_by = 1;
	return 0;
}


//parses the csv file in search of the passed shprogram name and try to load the shprogram.
//will return 0 if it was properly found and loaded, -1 otherwise.
int shprogram_search(char *shp_name, struct shp_info **shp_pointer)
{
	printf("searching for information on the shader program %s...\n", shp_name);
	
	//replaces the pointer by the one returned by malloc
	*shp_pointer = malloc(sizeof(struct shp_info));

	struct shp_info *current_shp = *shp_pointer;
	current_shp->name = malloc(sizeof(char) * 128);
	current_shp->vert_file_path = malloc(sizeof(char) * 128);
	current_shp->frag_file_path = malloc(sizeof(char) * 128);
	current_shp->used_by = 1;

	char *csv_string = csv_parse("assets/shprograms_index.csv", "shprogram_name", shp_name);
	if (csv_string == NULL)
	{
		fprintf(stderr, "Error: could not find the shprogram.\n");
		shprogram_unload(current_shp);
		return -1;
	}

	sscanf(csv_string, "%[^;];%u;%[^;];%[^\n]",
	current_shp->name,
	&current_shp->id,
	current_shp->vert_file_path,
	current_shp->frag_file_path);
	//load the shader and assign its OpenGL handle
	current_shp->handle = shprogram_load(current_shp);

	/*printf("shader_program_name=%s\nid=%u\nvertex_shader_path=%s\nfrag_shader_path=%s\nindex=%u\n",
	current_shp->name,
	current_shp->id, 
	current_shp->vert_file_path,
	current_shp->frag_file_path,
	current_shp->index);*/

	//shprogram_set_uniforms(current_shp);

	return 0;
}

//creates and loads a shader program. Will return its index if properly loaded, 
//-1 otherwise.
GLuint shprogram_load(struct shp_info *shp)
{
	GLint status;
	GLuint program;

	//this line MUST be BEFORE you load your shaders, unless you want to
	//loose days in debugging sessions
	program = glCreateProgram();

	GLuint vert_shader = shader_gl_load(shp->vert_file_path, GL_VERTEX_SHADER);
	GLuint frag_shader = shader_gl_load(shp->frag_file_path, GL_FRAGMENT_SHADER);

	glAttachShader(program, vert_shader);
	glAttachShader(program, frag_shader);


	glLinkProgram(program);

	glGetProgramiv(program, GL_LINK_STATUS, &status);
	if (status == GL_FALSE)
	{
		fprintf(stderr, "failed to link program, handle: %d\n", program);
		GLsizei length = 256;
		GLchar *log = malloc(length);
		glGetProgramInfoLog(program, length, &length, log);
		printf("OpenGL log:\n%s\n", log);
		free(log);
		//could be great to return the GLuint of a backup program, no?
	}
	else
	{
		fprintf(stdout, "successfully linked program, handle: %d\n", program);
	}

	glDetachShader(program, vert_shader);
	glDetachShader(program, frag_shader);

	return program;
}

void shprogram_set_uniforms(struct shp_info *shp)
{
	glUseProgram(shp->handle);

	//I should really find a way to not make unnecessary calls to get inexistant
	//uniforms locations depending on the shader used. Right now, it asks the
	//shader program for every uniforms locations and only the uniforms that the shader
	//has return a value, instead of -1. I mean, it works regardless, but it ain't pretty.

	//send the model, view and projection matrices to the bound shader program
	shp->uniform_model_matrix = glGetUniformLocation(shp->handle, "model_matrix");
	glUniformMatrix4fv(shp->uniform_model_matrix, 1, GL_FALSE, model_matrix);

	shp->uniform_view_matrix = glGetUniformLocation(shp->handle, "view_matrix");
	glUniformMatrix4fv(shp->uniform_view_matrix, 1, GL_FALSE, view_matrix);

	shp->uniform_mv_matrix = glGetUniformLocation(shp->handle, "mv_matrix");
	glUniformMatrix4fv(shp->uniform_mv_matrix, 1, GL_FALSE, mv_matrix);

	uniform_projection_matrix = glGetUniformLocation(shp->handle, "projection_matrix");
	glUniformMatrix4fv(uniform_projection_matrix, 1, GL_FALSE, projection_matrix);

	shp->uniform_object_color = glGetUniformLocation(shp->handle, "object_color");

	shp->uniform_offset = glGetUniformLocation(shp->handle, "offset");

	
	//The light position in the world. Need to find a good location for it.
	uniform_light_position = glGetUniformLocation(shp->handle, "light_position");
	struct vec3 light_pos;
	light_pos.x = 100.0f;
	light_pos.y = 200.5f;
	light_pos.z = 50.9f;
	glUniform3f(uniform_light_position, light_pos.x, light_pos.y, light_pos.z);


	GLint object_texture = glGetUniformLocation(shp->handle, "object_texture");


	//the shader now use texture unit 0.
	glUniform1i(object_texture, 0);

	glUseProgram(0);


	GLint result;
	glGetProgramiv(shp->handle, GL_ACTIVE_UNIFORMS, &result);
	printf("uniforms count: %d\n", result);
}


void shprogram_update_matrices(void)
{
	camera_update_view_matrix();

	//why does this have to be called in this function? If I set it in
	//the initialize function the program crashes.
	props_index = proplist_get_pointer();
	for (int i = 0; i < MAX_PROPS; i++) {
		if (props_index[i] == 0)
			continue;
		glUseProgram(props_index[i]->shprogram->handle);
		mat4_multiply(view_matrix, model_matrix, mv_matrix);
		
		glUniformMatrix4fv(props_index[i]->shprogram->uniform_mv_matrix, 1, GL_FALSE, mv_matrix);
		glUniformMatrix4fv(props_index[i]->shprogram->uniform_view_matrix, 1, GL_FALSE, view_matrix);
	}
	glUseProgram(0);
}


int shprogram_unload(struct shp_info *current_shp)
{
	free(current_shp->name);
	free(current_shp->vert_file_path);
	free(current_shp->frag_file_path);
	free(current_shp);

	return 0;
}
