#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "prop.h"
#include "../lists/proplist.h"
#include "../utils/mat4.h"
#include "../utils/num.h"



int prop_allocate(struct prop_info **prop)
{
	*prop = malloc(sizeof(struct prop_info));
	(*prop)->model_matrix = mat4_identity();
	(*prop)->rotation_matrix = mat4_identity(); //this will not be used for every prop! optimize?
	(*prop)->instances = malloc(sizeof(unsigned int)); //a pointer, so it should be shared amongst props of the same type (TODO)
	*(*prop)->instances = 1; //first instance of a prop
	(*prop)->name = malloc(sizeof(char) * PROP_MAX_NAME_LENGTH); //it was commented out. why?
	(*prop)->color_override = NULL;
	(*prop)->color = vec3(-1.0, -1.0, -1.0); //ugly, see if statement in renderer.c
	(*prop)->position = vec3_zero();
	(*prop)->velocity = vec3_zero();
	(*prop)->index = -1;
	(*prop)->visible = 1;
	(*prop)->exist = 1;
	(*prop)->rotation = quat(0, vec3_zero()); //should be converted to euler angles when saving a map
	(*prop)->map = 0;
	(*prop)->type = T_UNDEFINED;
	return 0;
}


int prop_deallocate(struct prop_info *prop)
{
	return 0;
}


//add the given vector to the prop's position.
void prop_move(struct prop_info *prop, struct vec3 position)
{
	mat4_translate(prop->model_matrix, position);

	prop->position = vec3_add(prop->position, position);

	if (prop->exist == 0 || prop->coll_model == NULL) //don't forget prop->exit == 0!
		return;

	for (int i = 0; i < prop->coll_model->vertex_count[MODEL_VERTEX_BUFFER]; i+=3) {
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i] += position.x;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+1] += position.y;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+2] += position.z;
	}
	for (int i = 0; i < prop->bbox2->vertices_count; i+=3) {
		prop->bbox2->vertices[i] += position.x;
		prop->bbox2->vertices[i+1] += position.y;
		prop->bbox2->vertices[i+2] += position.z;
	}
	//don't forget those points
	prop->bbox2->p1 = vec3_add(prop->bbox2->p1, position);
	prop->bbox2->p2 = vec3_add(prop->bbox2->p2, position);
	prop->bbox2->p3 = vec3_add(prop->bbox2->p3, position);
	prop->bbox2->p4 = vec3_add(prop->bbox2->p4, position);
	prop->bbox2->p5 = vec3_add(prop->bbox2->p5, position);
	prop->bbox2->p6 = vec3_add(prop->bbox2->p6, position);
	prop->bbox2->p7 = vec3_add(prop->bbox2->p7, position);
	prop->bbox2->p8 = vec3_add(prop->bbox2->p8, position);
}

//position the prop at the location of the given vector,
void prop_reset_pos(struct prop_info *prop, struct vec3 position)
{
	mat4_position(prop->model_matrix, position);

	//make a velocity vector between the old and new position
	struct vec3 old_pos = prop->position;
	prop->position = position;
	struct vec3 vel = vec3_subtract(old_pos, position);

	if (prop->exist == 0 || prop->coll_model == NULL) //don't forget prop->exit == 0!
		return;

	//apply that velocity vector to the collision model and the bounding box
	for (int i = 0; i < prop->coll_model->vertex_count[MODEL_VERTEX_BUFFER]; i+=3) {
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i] -= vel.x;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+1] -= vel.y;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+2] -= vel.z;
	}
	for (int i = 0; i < prop->bbox2->vertices_count; i+=3) {
		prop->bbox2->vertices[i] -= vel.x;
		prop->bbox2->vertices[i+1] -= vel.y;
		prop->bbox2->vertices[i+2] -= vel.z;
	}
	//don't forget those points
	prop->bbox2->p1 = vec3_subtract(prop->bbox2->p1, vel);
	prop->bbox2->p2 = vec3_subtract(prop->bbox2->p2, vel);
	prop->bbox2->p3 = vec3_subtract(prop->bbox2->p3, vel);
	prop->bbox2->p4 = vec3_subtract(prop->bbox2->p4, vel);
	prop->bbox2->p5 = vec3_subtract(prop->bbox2->p5, vel);
	prop->bbox2->p6 = vec3_subtract(prop->bbox2->p6, vel);
	prop->bbox2->p7 = vec3_subtract(prop->bbox2->p7, vel);
	prop->bbox2->p8 = vec3_subtract(prop->bbox2->p8, vel);
}

//scale the prop by the given vector.
void prop_scale(struct prop_info *prop, struct vec3 s)
{
	//vertices in OpenGL's buffer are not updated, we just scale the matrix
	mat4_scale(prop->model_matrix, s);

	if (prop->coll_model == NULL)
		return;

	for (int i = 0; i < prop->coll_model->vertex_count[MODEL_VERTEX_BUFFER]; i+=3) {
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i] *= s.x;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+1] *= s.y;
		prop->coll_model->vertex_data[MODEL_VERTEX_BUFFER][i+2] *= s.z;
	}
	for (int i = 0; i < prop->bbox2->vertices_count; i+=3) {
		prop->bbox2->vertices[i] *= s.x;
		prop->bbox2->vertices[i+1] *= s.y;
		prop->bbox2->vertices[i+2] *= s.z;
	}
	prop->bbox2->p1 = vec3_multiply(prop->bbox2->p1, s);
	prop->bbox2->p2 = vec3_multiply(prop->bbox2->p2, s);
	prop->bbox2->p3 = vec3_multiply(prop->bbox2->p3, s);
	prop->bbox2->p4 = vec3_multiply(prop->bbox2->p4, s);
	prop->bbox2->p5 = vec3_multiply(prop->bbox2->p5, s);
	prop->bbox2->p6 = vec3_multiply(prop->bbox2->p6, s);
	prop->bbox2->p7 = vec3_multiply(prop->bbox2->p7, s);
	prop->bbox2->p8 = vec3_multiply(prop->bbox2->p8, s);
}

//check if a given prop's bounding box plus a velocity vector intersect with any other bbox in the main
//prop list. parses the main list and add the current prop to a list if their bbox intersect.
//returns 1 if at least 1 prop is intersected, 0 otherwise.
int prop_check_colliding_bboxes(struct prop_info *prop, struct prop_info **props, unsigned int *count)
{
	int a = 0;
	struct prop_info ** props_index = proplist_get_pointer();
	for (int i = 0; i < MAX_PROPS; i++) {
		if (props_index[i] == 0 || !props_index[i]->exist)
			continue;
		if (strcmp(props_index[i]->name, "ENV_SKYBOX") == 0 ||
		strcmp(props_index[i]->name, "ENT_PLAYER") == 0) //TODO: could this be replaced with prop->exist?
			continue;
		//Just doing *count++ made the whole thing crash. I suppose I was in fact accessing
		//the value of memory next to the count variable, instead of incrementing it.
		if (bbox_collide(prop->bbox2, props_index[i]->bbox2, prop->velocity))
			props[a++] = props_index[i];
	}
	if (a) {
		*count = a;
		return 1;
	}
	*count = 0;
	return 0;
}

//check if a point is in props' bbox and add the props in a list if it's the case.
//A more precise solution would be to use coll_ray_collide_bbox(), since this method won't work if
//the object that requires this check is moving faster than the width of some of the bboxes. However,
//this one is more processor-friendly.
int prop_point_check_colliding_bboxes(struct vec3 point, struct prop_info **props, unsigned int *count)
{
	struct prop_info *p;
	int a = 0;
	struct prop_info ** props_index = proplist_get_pointer();
	for (int i = 0; i < MAX_PROPS; i++) {
		p = props_index[i];
		if (p == 0 || !p->exist) { continue; }
		if (bbox_point_collide(p->bbox2, point))
			props[a++] = p;
	}
	if (a) {
		*count = a;
		return 1;
	}
	*count = 0;
	return 0;
}

void prop_rotate(struct prop_info *p, struct quat q)
{
	mat4_from_quat(p->rotation_matrix, q);
}

void prop_random_color(struct prop_info *p)
{
	p->color.x = randf(0,1);
	p->color.y = randf(0,1);
	p->color.z = randf(0,1);
}

