#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "model.h"
#include "parser.h"
#include "../utils/vec3.h"
#include "../utils/vec2.h"




// --------------OBJ---------------- //

struct faced {
	int vert_index;
	int texture_coord_index;
	int normal_index;
};

static void obj_parse_vertex(char *line, struct vec3 *vertices_array, int index);
static void obj_parse_texture_coord(char *line, struct uv *textures_coords_array, int index);
static void obj_parse_normal(char *line, struct vec3 *normals_array, int index);
static void obj_parse_faces(char *line, struct faced *faces_array, int index);


//Loads an obj model and returns a pointer to a float
//array containing the parsed data. Returns NULL if the
//model could not be loaded.
int parser_load_obj(char *filepath, float *vertices_destination, unsigned int *vertices_count)
{
	FILE *file;
	//printf("Loading the model's OBJ file...\n");
	if ((file = fopen(filepath, "rb")) == NULL)
	{
		fprintf(stderr, "Error: the model's obj file could not be found or loaded\n");
		vertices_destination = NULL;
		return -1;
	}

	//right now only the vertices array and the faces array are used
	// "* 4" because x, y, z, w
	struct vec3 vertices_array[MODEL_MAX_VERTICES];
	struct uv texture_coords_array[MODEL_MAX_VERTICES];
	struct vec3 normals_array[MODEL_MAX_VERTICES];
	struct faced faces_array[MODEL_MAX_VERTICES];

	char line[OBJ_MAX_LINE_LENGTH];

	int count = 0;
	int vertices_array_index = 0;
	int texture_coords_array_index = 0;
	int normals_array_index = 0;
	int faces_array_index = 0;
	
	//get the necessary info out of the file and place it in the arrays
	while (fgets(line, OBJ_MAX_LINE_LENGTH, file) != NULL)
	{
		//this is where the program checks if the OBJ file contains too many vertices or not.
		if (count >= MODEL_MAX_VERTICES - 3)
		{
			fprintf(stderr, "Error: the model contains too many vertices and couldn't be loaded entirely\n");
			printf("DATA!: %d\n", count);
			vertices_destination = NULL;
			return -1;
		}

		if (line[0] == 'v' && line[1] == ' ')
		{
			obj_parse_vertex(line, vertices_array, vertices_array_index);
			vertices_array_index++;
		}
		else if (line[0] == 'v' && line[1] == 't')
		{
			obj_parse_texture_coord(line, texture_coords_array, texture_coords_array_index);
			texture_coords_array_index++;
		}
		else if (line[0] == 'v' && line[1] == 'n')
		{
			obj_parse_normal(line, normals_array, normals_array_index);
			normals_array_index++;
		}
		else if (line[0] == 'f' && line[1] == ' ')
		{
			obj_parse_faces(line, faces_array, faces_array_index);
			faces_array_index += 3;
			count+=3;
		}
	}

	//printf("Duplicating the vertices...\n");
	//duplicates the vertices based on the faces array. The program doesn't support vertex indexing yet.
	int data_index = 0;
	for (int i = 0; i < faces_array_index; i++)
	{
		struct faced current_face = faces_array[i];

		//get the vertices data and upload it to the vertices destination
		struct vec3 point = vertices_array[current_face.vert_index - 1];
		// - 1 because OBJ indexing starts at 1, and array indexing at 0

		vertices_destination[data_index++] = point.x;
		vertices_destination[data_index++] = point.y;
		vertices_destination[data_index++] = point.z;
		vertices_destination[data_index++] = 1.0f;

		struct vec3 normal = normals_array[current_face.normal_index - 1];
		vertices_destination[data_index++] = normal.x;
		vertices_destination[data_index++] = normal.y;
		vertices_destination[data_index++] = normal.z;
		vertices_destination[data_index++] = 1.0f;

		struct uv coords = texture_coords_array[current_face.texture_coord_index - 1];
		vertices_destination[data_index++] = coords.u;
		vertices_destination[data_index++] = coords.v;

	}
	*vertices_count = data_index/* / 4*/;

	return 0;
}


static void obj_parse_vertex(char *line, struct vec3 *vertices_array, int index)
{
	struct vec3 vector;
	sscanf(line, "%*s %f %f %f", &vector.x, &vector.y, &vector.z);
	vertices_array[index] = vector;
}
static void obj_parse_texture_coord(char *line, struct uv *texture_coords_array, int index)
{
	/*float x, y;
	sscanf(line, "%*s %f %f", &x, &y);
	texture_coords_array[index++] = x;
	texture_coords_array[index++] = y;*/
	struct uv coords;
	sscanf(line, "%*s %f %f", &coords.u, &coords.v);
	texture_coords_array[index] = coords;
}
static void obj_parse_normal(char *line, struct vec3 *normals_array, int index)
{
	struct vec3 vector;
	sscanf(line, "%*s %f %f %f", &vector.x, &vector.y, &vector.z);
	normals_array[index] = vector;
	//printf("NORMAL %f %f %f\n", vector.x, vector.y, vector.z);
}
static void obj_parse_faces(char *line, struct faced *faces_array, int index)
{
	struct faced face_1, face_2, face_3;
	sscanf(line, "%*s %d/%d/%d %d/%d/%d %d/%d/%d",
	&face_1.vert_index,
	&face_1.texture_coord_index,
	&face_1.normal_index,
	&face_2.vert_index,
	&face_2.texture_coord_index,
	&face_2.normal_index,
	&face_3.vert_index,
	&face_3.texture_coord_index,
	&face_3.normal_index);

	faces_array[index++] = face_1;
	faces_array[index++] = face_2;
	faces_array[index++] = face_3;
}



// ---------------RAW----------------- //
// outdated, as the VBO no expect vertex normals with the vertices, which the raw format doesn't provide.


int parser_load_raw(char *filepath, float *vertices_destination, unsigned int *vertices_count)
{
	FILE *model_file;
	float pos_x, pos_y, pos_z, pos_w;
	int count = 0;
	unsigned int size = sizeof(float) * 4;

	//check if the passed filepath can be opened
	if ((model_file = fopen(filepath, "rb")) == NULL)
	{
		fprintf(stderr, "Error: could not open the model's file.\n");
		return 1;
	}

	//while there's still vertices in the file and still space in the array...
	while (fscanf(model_file, "%f %f %f %f", &pos_x, &pos_y, &pos_z, &pos_w) != EOF && (count + 4) < MODEL_MAX_VERTICES)
	{
		//find a way to get this variable out
		size = size + sizeof(float) * 4;

		vertices_destination[count++] = pos_x;
		vertices_destination[count++] = pos_y;
		vertices_destination[count++] = pos_z;
		vertices_destination[count++] = pos_w;

		/*print the last vertex added 
		for (int i = count - 4; i < count; i++)
		{
			printf("%f\t", vertices_destination[i]);
			if (i == count - 1)
			{
				printf("\n");
			}
		}*/
	}
	fclose(model_file);

	*vertices_count = count;
	
	if (count + 4 >= MODEL_MAX_VERTICES)
	{
		fprintf(stderr, "Error: the model contains too many vertices and couldn't be loaded entirely\n");
		vertices_destination = NULL;
		return 1;
	}
	return 0; 
}
