#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "loader.h"
#include "model.h"
#include "shprogram.h"
#include "texture.h"
#include "vao.h"
#include "vbo.h"
#include "../utils/mat4.h"
#include "../lists/proplist.h"
#include "../game/player.h"
#include "../utils/conf.h"
#include "../ui/button.h"


static int duplicate_asset(struct prop_info*, struct prop_info*, struct vec3, enum object_type, char *);
static struct model* load_model(char *);
static struct bbox_info* load_bbox(struct model*, enum object_type);
static struct shp_info* load_shprogram(struct model*);
static struct texture_info* load_texture(struct model*);


int loader_load_asset_new(char *name, struct vec3 position, enum object_type type)
{
	//careful with loading same props with different types! variables are not copied correctly,
	//and you could end up with invisible props, weird collision models, etc.
	struct prop_info *prop;
	prop_allocate(&prop);

	struct prop_info *existing_prop;
	//if the prop has already been loaded...
	if ((existing_prop = proplist_get_prop(name)) != NULL)
		return duplicate_asset(prop, existing_prop, position, type, name);

	prop->name = name;
	prop->position = position;

	//load the model. collision model if needed, shprogram and texture
	load_components(prop, type);

	int id = proplist_add(prop);
	//the position needs to be set to 0 as it's used in prop_reset_pos()
	prop->position = vec3_zero();
	//don't MOVE the prop, reset it's position
	prop_reset_pos(prop, position);
	//not needed anymore as the bboxes are moved in prop_move() and prop_reset_pos()
	//bbox_to_world_space(prop->bbox2, prop->model_matrix);

	switch (type) {
	case SKYBOX:
		proplist_set_skybox(prop);
		prop->exist = 0;
		break;
	case PLAYER:
		proplist_set_player(prop);
		player_set_position(position);
		prop->exist = 1;
		prop->visible = 0; //you want to be able to see
		break;
	case T_ENTITY: //entities are rendered and checked for collisions differently
		prop->visible = 0;
		prop->exist = 0;
		break;
	case T_OPLAYER:
		prop->exist = 0;
		break;
	case T_UI:
		prop->exist = 0;
		break;
	case T_MAP_PROP:
		prop->map = 1; //the prop comes from the map
		break;
	default:
		break;
	}
	prop->type = type;
	prop->index = id;
	return id;
}

int load_components(struct prop_info *prop, enum object_type type)
{
//---------model work-------------
	prop->model = load_model(prop->name);

//-----collision model work-------
	if (type == PROP || type == T_MAP_PROP || type == T_UI) //T_UI has the exist flag set to 0, so they won't collide anyway
		prop->coll_model = load_model(prop->name);
	else
		prop->coll_model = NULL;

//-------bounding box work--------
	prop->bbox2 = load_bbox(prop->model, type);

//--------shprogram work----------
	prop->shprogram = load_shprogram(prop->model);

//---------texture work-----------
	prop->texture = load_texture(prop->model);
	//printf("%s\n", texture->name); why do I get a segmentation fault here?

//--------------------------------
	return 0;
}

int duplicate_asset(struct prop_info *prop, struct prop_info *existing_prop, struct vec3 position, enum object_type type, char *name)
{

	struct model *coll_model;
	struct bbox_info *bbox2;

	prop->name = existing_prop->name;
	prop->model = existing_prop->model;
	prop->shprogram = existing_prop->shprogram;
	prop->texture = existing_prop->texture;
	prop->instances = existing_prop->instances;
	prop->exist = existing_prop->exist;
	prop->visible = existing_prop->visible;
	prop->type = existing_prop->type;

	*prop->instances += 1;
	printf("There is now %u instances of this model.\n", *prop->instances);

	//creating a new bbox2
	bbox_allocate(&bbox2);
	prop->bbox2 = bbox2;
	vao_create(&bbox2->vao_handle);
	vao_bind(bbox2->vao_handle);
	if (type != PLAYER && type != T_OPLAYER) //this line made the game crash in multiplayer, I commented it
		bbox_get(existing_prop->model, bbox2);
	else {
		bbox_generate(bbox2, 10, 10, 10);
	}
	vao_unbind();

	//creating a new coll_model
	model_allocate(&coll_model);
	vao_create(&coll_model->vao_handle);
	vao_bind(coll_model->vao_handle);
	if (model_load_new(name, coll_model) < 0) {
		fprintf(stderr, "Error: failed to load collision model\n");
		model_unload(coll_model);
		return -1;
	}
	vao_unbind();
	prop->coll_model = coll_model;

	//---

	prop_move(prop, position);
	prop->position = position; //don't forget this!
	//bbox_to_world_space(prop->bbox2, prop->model_matrix);
	int id = proplist_add(prop);
	prop->index = id;
	return id;
}


struct model* load_model(char *name)
{
	struct model *model;
	model_allocate(&model);

	vao_create(&model->vao_handle);
	vao_bind(model->vao_handle);

	//try to load the model
	if (model_load_new(name, model) < 0) {
		fprintf(stderr, "Error: failed to load model\n");
		model_unload(model);
		return NULL;
	}
	vao_unbind();
	return model;
}

struct bbox_info* load_bbox(struct model* model, enum object_type type) {

	struct bbox_info *bbox;
	bbox_allocate(&bbox); //error handling here
	vao_create(&bbox->vao_handle);
	vao_bind(bbox->vao_handle);
	if (type != PLAYER)
		bbox_get(model, bbox);
	else
		bbox_generate(bbox, 5, 8, 5);
	vao_unbind();
	return bbox;
}

struct shp_info* load_shprogram(struct model* model)
{
	struct shp_info *shp, *existing_shp;

	if ((existing_shp = proplist_get_shp(model->shprogram_name)) != NULL) {
		existing_shp->used_by++;
		printf("reusing shader %s, there are now %d sorts of props using it\n", existing_shp->name, existing_shp->used_by);
		return existing_shp;
	}

	shprogram_search(model->shprogram_name, &shp);
	//shprogram_allocate(&shp); //TODO
	shprogram_set_uniforms(shp);
	return shp;
}

struct texture_info* load_texture(struct model* model)
{
	struct texture_info *texture;
	if(model->texture[0] == ' ')
		return NULL;
	
	texture_allocate(&texture);
	if (texture_search(model->texture, texture) < 0) {
		fprintf(stderr, "Error: could not load the texture\n");
		texture_deallocate(texture);
		return NULL;
	}

	//load the texture, providing the texture struct containing the necessary info
	if (texture_load(texture) < 0) {
		fprintf(stderr, "Error: could not load the texture\n");
		texture_deallocate(texture);
		return NULL;
	}

	texture_bind(texture);
	texture_unbind();

	return texture;
}

//remove a prop from the proplist, and unload
//it if there is no other instances of it
int loader_unload(struct prop_info *p)
{
	//this function is a huuuge TODO
	if (p->instances[0] > 1) {
		printf("unload: prop used multiple times\n");
	}

	proplist_remove(p->index);

	return 0;
}
