#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "player.h"
#include "camera.h"
#include "../utils/states.h"
#include "../lists/proplist.h"
#include "../utils/etc.h"
#include "collision.h"
#include "../loader/prop.h"
#include "../net/clt.h"
#include "../utils/quat.h"

//a variable player_weight would be good too
//please do a struct
static struct vec3 player_position;
static struct vec3 old_position;
static float player_gravity;
static float current_gravity_vector;
static struct vec3 player_max_velocity;
static struct vec3 player_velocity;
static struct vec3 player_wishdir; //the direction where the user wants to go
static float player_speed;
static float friction;
static float air_accel;
static float max_air_speed;
static int max_health = 100;
static int health;

static float sv_accel_ground; //the acceleration on the ground
static float sv_max_air_speed; //the acceleration in the air
static float sv_accel_air; //the max speed the user can move at in the air (not counting strafing)

static int airborne;
static int noclip;
static int ghost;
static int jumped;

static int mult;

static struct prop_info **colliding_props;

static void update(void);
static void air_strafe(void);



int player_initialize(void)
{
	air_accel = 10.0f;
	max_air_speed = 1;
	sv_accel_ground = 0.004f;
	sv_max_air_speed = 0.15f;
	sv_accel_air = 0.004f;

	player_position = vec3_zero();
	old_position = vec3_zero();
	player_velocity = vec3_zero();
	player_wishdir = vec3_zero();

	player_max_velocity = vec3_zero();
	player_max_velocity.y = -2.95f;

	player_gravity = 0.0009f;
	player_speed = 0.075f;
	friction = 0.0001; //uuh, overriden in update_player(), hihi
	airborne = 0;
	noclip = 0;
	ghost = 0;
	jumped = 0;
	health = max_health;

	colliding_props = malloc(sizeof(struct prop_info *) * MAX_PROPS);

	return 0;
}

void player_move_left(void)
{
	player_wishdir.x += sin(degrees_to_radians(camera_get_yaw()));
	player_wishdir.z -= cos(degrees_to_radians(camera_get_yaw()));
}
void player_move_right(void)
{
	player_wishdir.x -= sin(degrees_to_radians(camera_get_yaw()));
	player_wishdir.z += cos(degrees_to_radians(camera_get_yaw()));
}

void player_move_forward(void)
{
	player_wishdir.x += cos(degrees_to_radians(camera_get_yaw()));
	player_wishdir.z += sin(degrees_to_radians(camera_get_yaw()));
}

void player_move_backward(void)
{
	player_wishdir.x -= cos(degrees_to_radians(camera_get_yaw()));
	player_wishdir.z -= sin(degrees_to_radians(camera_get_yaw()));
}

void player_move_up(void)
{
	if (noclip) {
		player_velocity.y = 0.02f * extern_delta_time;
		return;
	}

	//if the player is on the ground and hasn't jumped, he's elligible to jump again
	if (!airborne && !jumped) {
		airborne = 1;
		player_velocity.y = 0.35f;
	}
	jumped = 1;
}

void player_move_down(void)
{
	if (noclip)
		player_velocity.y = -0.02f * extern_delta_time;
	//down();
}

struct vec3 player_get_position(void)
{
	return player_position;
}

void player_set_position(struct vec3 position)
{
	player_position = position;
	update();
}

static void update(void)
{
	//I don't get why, but placing those lines here removes the "lagginess" of the props supposed
	//to follow the player's position.
	prop_reset_pos(proplist_get_skybox(), player_position);
	prop_reset_pos(proplist_get_player(), player_position);

	int collision = 0;

	player_velocity.x *= friction; //move this to air_strafe()?
	player_velocity.z *= friction;

	struct collision_info coll_info;

	
	if (!ghost) {
		unsigned int count;
		if (prop_check_colliding_bboxes(proplist_get_player(), colliding_props, &count)) {
			//printf("collision happened with %u objects\n", count);
			coll_info.colliding_props = colliding_props;
			coll_info.colliding_props_count = count;
			if (collision_check(&coll_info, &player_position, &player_velocity, proplist_get_player()))
				//it's the collision function that decides wether the player is
				//airborne or not. It shouldn't be a problem, since the player should
				//almost always be in another model's bounding box at all times, and
				//jumping sets the player's airborne state to true.
				airborne = coll_info.airborne;
			collision = 1;
		}
	}


	//update the position here since the collision function is never called if there weren't any collisions
	//with any model's bounding box
	if (!collision || ghost) {
		player_position.x += player_velocity.x;
		player_position.y += player_velocity.y;
		player_position.z += player_velocity.z;
	}
	
	proplist_get_player()->position = player_position;
	proplist_get_player()->velocity = player_velocity;

	//since I can't make Y-elongated ellipsoids for now, I translate
	//the camera a bit to make it look like the player is not the size of
	//a gnome
	camera_set_position(vec3_add(player_position, vec3(0, 3.0f, 0)));


	float yaw = degrees_to_radians(camera_get_yaw() - 90); //why -90 tho
	struct quat q = quat(yaw, vec3(0, 1, 0));
	//struct vec3 rot = camera_get_front();
	if (extern_is_client || extern_is_server)
		clt_out_player(&player_position, &q); // inform the server of the position and rotation of the player
}

//prepare some variables before updating the player's position
void player_update_position(void)
{
	if (airborne) {
		mult = 1;
		friction = 1;
	} else {
		mult = 10;
		friction = 0.85;
	}
	if (!noclip) {
		current_gravity_vector = player_gravity * extern_delta_time;
		player_velocity.y -= current_gravity_vector;
		if (player_velocity.y < player_max_velocity.y)
			player_velocity.y = player_max_velocity.y;
	}

	
	//weird: sometimes the position of the bounding box can be very slightly misaligned (3.999999 instead
	//of 4) each time the player hits something. It adds up.
	bbox_move(proplist_get_player()->bbox2, vec3_subtract(player_position, old_position));

	old_position = player_position;
	air_strafe();
	update();
}


//this is called when the user releases the jump button
void player_jump_stop(void)
{
	jumped = 0;
	if (noclip) {
		player_velocity.y = 0;
	}
}

//this is called when the user releases the crouch button
void player_crouch_stop(void)
{
	if (noclip) {
		player_velocity.y = 0;
	}
}

void player_toggle_noclip(void)
{
	player_velocity.y = 0;
	noclip = !noclip;
}

void player_toggle_collisions(void)
{
	ghost = !ghost;
}

float player_get_current_gravity(void)
{
	return current_gravity_vector;
}

void player_push(struct vec3 vel)
{
	player_velocity = vec3_add(player_velocity, vel);
	airborne = 1; //needed if the player shoots below him
}

//movement code for the player in the air. This essentially is the movement used in quake games or
//the source engine.
static void air_strafe(void)
{
	if (vec3_length(player_wishdir) == 0) { return; } //if the player doesn't want to move, return.
	player_wishdir = vec3_normalize(player_wishdir);

	float accel_ground = sv_accel_ground * extern_delta_time;

	if (!airborne) { //if the player isn't airborne, just move the user where it wants to go
		player_velocity = vec3_add(player_velocity, vec3_multiply_float(player_wishdir, accel_ground));
		player_wishdir = vec3_zero();
		return;
	}


	//projected length of the wish direction onto the velocity vector
	float proj_length = vec3_dot(player_velocity, player_wishdir);

	//if the length is bigger than the max allowed air length, don't add anything and return.
	if (proj_length > sv_max_air_speed) {
		player_wishdir = vec3_zero();
		return;
	}

	//calculate air acceleration
	float accel_vel = sv_accel_air * extern_delta_time;

	//clip the acceleration if it would be, coupled with the projected length, bigger than the max allowed speed.
	if (proj_length + accel_vel > sv_max_air_speed)
		accel_vel = sv_max_air_speed - proj_length;

	//add the calculated wishdir to the player.
	player_velocity = vec3_add(player_velocity, vec3_multiply_float(player_wishdir, accel_vel));

	//should be freed here
	player_wishdir = vec3_zero();
}

int player_get_hp(void)
{
	return health;
}

void player_damage(int dmg)
{
	health -= dmg;
	if (health <= 0) {
		player_position = vec3(0,50,0);
		player_velocity = vec3(0,0,0);
		health = max_health;
	}
}
