#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cmd.h"
#include "../utils/str.h"
#include "../map/map2.h"
#include "../map/write.h"
#include "../utils/states.h"
#include "../utils/io.h"
#include "../net/clt.h"

#include "cmd_exec.h"

#define CMD_MAX 8 //the number of commands in the program.
#define CMD_MAX_ARGS 2 //the max number of arguments a command can have
#define CMD_MAX_ARG_MEM 64 //the memory allocated to each argument

//this argument array is extern and shared with cmd_exec.c
void *args[CMD_MAX_ARGS] = {0};

struct cmd {
	char *name;		//the name of the command (what the user has to type to execute it)
	char *format;		//the format to extract the commands eventual arguments (ex: "%s %s %d")
	int argc;		//the number of arguments the command needs
	char *hformat;		//a human readable version of the format (displayed on error)
	char *man;		//a short description of the command, used by the command "man"
	int (*func)(void);	//the function that will be executed when the command is entered
};
static struct cmd cmds[CMD_MAX] = {0};

static int parse(char *);
static int parse_args(char *, struct cmd *);

//fill the command array with all the commands that the program supports.
void cmd_initialize(void)
{
	cmds[0].name = "map_write";
	cmds[0].man = "Writes the current map to it's file, saving any changes done to it.\
	The game must be launched in edit mode.";
	cmds[0].format = NULL;
	cmds[0].argc = 0;
	cmds[0].hformat = NULL;
	cmds[0].func = cmd_map_write;
	//
	cmds[1].name = "map";
	cmds[1].man = "Passes the following string to the map loader.";
	cmds[1].format = "%[^\n]"; //the rest of the string is sent to the map loader
	cmds[1].argc = 1;
	cmds[1].hformat = "map <command>";
	cmds[1].func = cmd_map;
	//
	cmds[2].name = "q";
	cmds[2].man = "Quits the game.";
	cmds[2].format = NULL;
	cmds[2].argc = 0;
	cmds[2].hformat = NULL;
	cmds[2].func = cmd_q;
	//
	cmds[3].name = "retry";
	cmds[3].man = "Reconnects to the current server.";
	cmds[3].format = NULL;
	cmds[3].argc = 0;
	cmds[3].hformat = NULL;
	cmds[3].func = cmd_retry;
	//
	cmds[4].name = "version";
	cmds[4].man = "Prints version informations.";
	cmds[4].format = NULL;
	cmds[4].argc = 0;
	cmds[4].hformat = NULL;
	cmds[4].func = cmd_version;
	//
	cmds[5].name = "man";
	cmds[5].man = "Prints the following command's manual.";
	cmds[5].format = "%s";
	cmds[5].argc = 1;
	cmds[5].hformat = "man <command>";
	cmds[5].func = cmd_man;
	//
	cmds[6].name = "clear";
	cmds[6].man = "Clears the chat.";
	cmds[6].format = NULL;
	cmds[6].argc = 0;
	cmds[6].hformat = NULL;
	cmds[6].func = cmd_clear;
	//
	cmds[7].name = "kill";
	cmds[7].man = "Kills yourself.";
	cmds[7].format = NULL;
	cmds[7].argc = 0;
	cmds[7].hformat = NULL;
	cmds[7].func = cmd_kill;
}

//parse and execute the given command encoded in c. c must have a newline followed by a
//null terminator in order to work properly!
void cmd_exec(char *c)
{
	char cmd_str[CMD_MAX_LENGTH] = {0};
	int l = strlen(c);
	//copy the given command to another string (because c can contain data after a null
	//pointer from older commands and this can cause old arguments to still be read)
	//TODO: make keyboard.c clean it's char buffer
	memcpy(cmd_str, c, l <= CMD_MAX_LENGTH ? l : CMD_MAX_LENGTH);

	//isolate the 1st word (must be the command)
	//info: from the manpage: strtok is not thread-safe
	if (strtok(cmd_str, "\n ") == NULL) {
		//putting \n in the delimiters is for if the cmd has no arguments following
		//it. if not put here, parse() cannot find the command.
		prnte("Error: no commands given\n");
		return;
	}

	//identify the entered command
	int cmd = parse(cmd_str);
	if (cmd == -1) {
		prnte("Error: command not recognized: %s\n", cmd_str);
		return;
	}

	//if the command takes no arguments, no need to do the work below. execute it and return.
	if (cmds[cmd].argc == 0) {
		cmds[cmd].func();
		return;
	}

	//create a char pointer pointing to the command arguments
	//I'm simply incrementing the cmd pointer by it's own length + 1 to
	//get past the null char that strtok put into it
	char *args_str = cmd_str + strlen(cmd_str) + 1;

	//allocate memory for all possible arguments
	for (int i = 0; i < CMD_MAX_ARGS; i++)
		args[i] = calloc(1, CMD_MAX_ARG_MEM);

	//identify and retrieve the arguments for the command
	if (parse_args(args_str, &cmds[cmd]) < 0) {//if wrong arguments
		prnte("usage: %s\n", cmds[cmd].hformat);
		for (int i = 0; i < CMD_MAX_ARGS; i++)
			free(args[i]);
		return;
	}

	if (cmds[cmd].func() == -2) //is this check necessary now?
		prnte("%s\n", cmds[cmd].hformat);

	for (int i = 0; i < 2; i++)
		free(args[i]);

}

//go through the command array to find the command whose name
//matches the given string.
//return the index of the command in the array if found, -1 otherwise.
static int parse(char *cmd_str)
{
	for (int i = 0; i < CMD_MAX; i++) {
		if (cmds[i].name == NULL) { continue; }
		if (strcmp(cmds[i].name, cmd_str) == 0) {
			return i;
		}
	}
	return -1;
}

//parse given arguments in string and verify that there is enough
//arguments for the command about to be executed.
//return 0 if no errors, -1 if there isn't enough arguments in string
static int parse_args(char *string, struct cmd *cmd)
{
	if (sscanf(string, cmd->format,  args[0], args[1]) != cmd->argc) {
		return -1;
	}
	return 0;
}

//show the manual of the given command.
void cmd_show_manual(char *c)
{
	int i;
	if ((i = parse(c)) < 0) {
		prnte("Error: no commands exist under the name %s\n", c);
		return;
	}
	prnti("%s\n", cmds[i].man);
	if (cmds[i].hformat != NULL)
		prnti("%s\n", cmds[i].hformat);
}
