#include <stdio.h>
#include <string.h>
#include <math.h>
#include "camera.h"
#include "../utils/vec4.h"
#include "../utils/vec3.h"
#include "../loader/shprogram.h"
#include "../utils/mat4.h"
#include "../utils/states.h"
#include "../utils/etc.h"
#include <GL/freeglut.h>


static float *projection_matrix;
static float *view_matrix;

static struct vec3 camera_coordinates;
static struct vec3 camera_front;
static struct vec3 camera_up;

static float camera_pitch;
static float camera_yaw;

static float camera_speed;


void camera_initialize(void)
{
	camera_speed = 0.015f;

	camera_pitch = 0.0f;
	camera_yaw = -90.0f;

	camera_coordinates.x = 0.0f;
	camera_coordinates.y = 0.0f;
	camera_coordinates.z = 0.0f;

	camera_front.x = 0.0f;
	camera_front.y = 0.0f;
	camera_front.z = -1.0f;

	camera_up.x = 0.0f;
	camera_up.y = 1.0f;
	camera_up.z = 0.0f;

	view_matrix = mat4_identity();
	camera_update_view_matrix();
	projection_matrix = mat4_projection(80.0f, 0.1f, 600.0f, 16.0f / 9.0f);
}

//Updates the view matrix. This should be called whenever the camera moved.
void camera_update_view_matrix(void)
{
	//we're adding the camera coordinates with the camera front, because
	//we always want that the camera looks ahead of itself. If that makes
	//sense.
	mat4_look_at(view_matrix, camera_coordinates, vec3_add(camera_coordinates, camera_front), camera_up);
}



//Checks if the mouse has moved. Now that I think of it, it's not
//useful while using FreeGLUT, but might be if I switch to GLFW(?)
void camera_check_movement(int x, int y)
{
	int center_x = extern_window_width / 2;
	int center_y = extern_window_height / 2;

	int delta_x = x - center_x;
	int delta_y = y - center_y;

	//if the cursor moved...
	if (delta_x != 0 || delta_y != 0)
	{
		camera_rotate(delta_x, delta_y);
		glutWarpPointer(center_x, center_y);
		shprogram_update_matrices();
	}
}

//rotates the camera based on the distances that the mouse travelled.
void camera_rotate(int delta_x, int delta_y)
{
	float mouse_speed = 0.06f;

	camera_yaw += mouse_speed * delta_x;
	camera_pitch -= mouse_speed * delta_y;

	//to avoid the camera doing weird things (gimball lock), restrict
	//its pitch to be between 89 and -89 degrees.
	if (camera_pitch > 89.0f)
	{
		camera_pitch = 89.0f;
	}
	if (camera_pitch < -89.0f)
	{
		camera_pitch = -89.0f;
	}

	if (camera_yaw < 0.0f) {
		camera_yaw = 360 + camera_yaw;
	}
	if (camera_yaw > 360.0f) {
		camera_yaw = 0 + (camera_yaw - 360);
	}

	camera_front.x = cos(degrees_to_radians(camera_yaw)) * cos(degrees_to_radians(camera_pitch));
	camera_front.y = sin(degrees_to_radians(camera_pitch));
	camera_front.z = sin(degrees_to_radians(camera_yaw)) * cos(degrees_to_radians(camera_pitch));
}

float * camera_get_projection_matrix_pointer(void)
{
	return projection_matrix;
}

float * camera_get_view_matrix_pointer(void)
{
	return view_matrix;
}

struct vec3 camera_get_front(void)
{
	return camera_front;
}

struct vec3 camera_get_up_axis(void)
{
	return camera_up;
}

void camera_set_position(struct vec3 position)
{
	camera_coordinates = position;
}

float camera_get_yaw(void)
{
	return camera_yaw;
}

float camera_get_pitch(void)
{
	return camera_pitch;
}

struct vec3 camera_get_position(void)
{
	return camera_coordinates;
}
