#include <stdio.h>
#include <stdlib.h>

#include "proj.h"
#include "../lists/proplist.h"
#include "../game/coll.h"
#include "../game/player.h"
#include "../utils/states.h"
#include "../net/clt.h"
#include "../net/oplayers.h"

static struct proj **projs; //the array of projectiles
static int proj_c;
static struct prop_info *rocket; //the default prop used for rockets
static struct prop_info *explosion; //the default prop used for explosions
static struct coll_info ci; //the collision informations shared with all the projectiles

static int add_proj(struct proj *);
static void check_coll(struct proj *);
static int check_ttl(struct proj *);
static void explode(struct proj *);

//allocate memory and pre-load projectile models.
int proj_initialize(void)
{
	projs = calloc(MAX_PROJECTILES, sizeof(struct proj));
	//ci.coll_props = malloc(sizeof(struct prop_info *) * MAX_PROPS);
	ci.t = 1;
	rocket = proplist_get_by_index(loader_load_asset_new("ENT_DEF_PROJ", vec3(0,0,0), T_ENTITY));
	explosion = proplist_get_by_index(loader_load_asset_new("ENT_DEF_EXPLOSION", vec3(0,0,0), T_ENTITY));
	proj_c = 0;
	return 0;
}

//loads a new projectile.
int proj_load(struct vec3 pos, struct vec3 ori, float speed, int owner, enum P_TYPE pt)
{
	vec3_normalize(ori);
	struct proj *p = malloc(sizeof(struct proj));
	p->position = pos;
	p->velocity = vec3_multiply_float(ori, speed);
	p->model_matrix = mat4_identity();
	p->birth = extern_current_time;
	p->speed = speed;
	p->damage = 0;
	p->owner = owner;
	switch (pt) {
	case P_OROCKET: //ugh
		p->exist = 1;
		p->explosion_radius = 15;
		p->knockback = 0.75;
		p->prop = rocket;
		p->ttl = 5000;
		p->damage = 49;
		break;
	case P_ROCKET:
		p->exist = 1;
		p->explosion_radius = 15;
		p->knockback = 0.75;
		p->prop = rocket;
		p->ttl = 5000;
		p->damage = 10;
		break;
	case P_EXPLOSION:
		p->exist = 0;
		p->explosion_radius = 0;
		p->knockback = 0;
		p->prop = explosion;
		p->ttl = 50;
	}
	if (pt == P_ROCKET) { //pretty ugly, find a better way
		if (extern_is_server || extern_is_client)
			clt_out_rocket(p);
	}

	p->index = add_proj(p);
	mat4_position(p->model_matrix, p->position);
	//printf("add at %d\n", p->index);
	return 0;
}

int proj_unload(struct proj* p)
{
	projs[p->index] = NULL;
	proj_c--;
	free(p);
	return 0;
}

//update all projectiles.
void proj_update_all(void)
{
	if (proj_c == 0)
		return;
	int c = 0;
	struct proj *p;
	for (int i = 0; i < MAX_PROJECTILES; i++) {
		p = projs[i];
		if (p == 0 || p == NULL) { continue; }
		if (c == proj_c) { return; }

		//check if they need to be deleted
		if (check_ttl(p)) { continue; }

		//should the projectile interact with the world?
		if (!p->exist) { continue; }

		//update their position and check if they collide
		check_coll(p);
		c++;
	}
}

static int add_proj(struct proj *p)
{
	if (proj_c == MAX_PROJECTILES) { return -1; }

	for (int i = 0; i < MAX_PROJECTILES; i++) {
		if (projs[i] == 0 || projs[i] == NULL) {
			projs[i] = p;
			proj_c++;
			return i;
		}
	}
	return -1;
}

struct proj ** proj_get_ptr(void)
{
	return projs;
}

//check if the projectile enters in collision with map props. unload it if it's the case
static void check_coll(struct proj *p)
{
	//(this isn't a good thing to do; the server should handle all of this)
	
	if (prop_point_check_colliding_bboxes(vec3_add(p->position, p->velocity),
	ci.coll_props, &ci.coll_props_count)) { //projectile collides with some props' bboxes

	//check if one of the bboxes belong to our player
	for (int i = 0; i < ci.coll_props_count; i++) {
		//if it's the case, and it's not one of you own projectiles, make it explode.
		if (proplist_get_player()->index == ci.coll_props[i]->index && p->owner != -1) {
			explode(p);
			proj_load(p->position, vec3_zero(), 0, -1, P_EXPLOSION);
			proj_unload(p);
			ci.t = 1; //revert t so following collisions don't use t from the old one
			return;
		}
	}
	//else, if it's map geometry

		ci.position = p->position;
		ci.velocity = p->velocity;
		if (coll_point_collide(&ci)) { //projectile collides with a prop's mesh
			p->position = ci.collision_point;
			mat4_position(p->model_matrix, p->position);
			explode(p);
			proj_load(p->position, vec3_zero(), 0, -1, P_EXPLOSION);
			proj_unload(p);
			ci.t = 1; //revert t so following collisions don't use t from the old one
			return;
		}
	}
	//if we get here, no collision happened
	p->position = vec3_add(p->position, p->velocity);
	mat4_position(p->model_matrix, p->position);
}

//check if the projectile should get unloaded due to expired time.
static int check_ttl(struct proj *p)
{
	if (extern_current_time - p->birth > p->ttl) {
		proj_unload(p);
		return 1;
	}
	return 0;
}

//explode the projectile, and apply knockback and damage to the player if he's close enough.
static void explode(struct proj *p)
{
	struct vec3 ppos = vec3_add(player_get_position(), vec3(0,2,0));
	float dist = vec3_distance(p->position, ppos);
	if (dist > p->explosion_radius) { return; } //not close enough

	//if the owner isn't -1 it means it's from an online player, and if it's from the same
	//team, don't deal any damage to the player
	if (p->owner != -1 && oplayers_get_team_from_pid(p->owner) == clt_get_team())
		return;

	struct vec3 dist_vector = vec3_normalize(vec3_subtract(ppos, p->position));

	//if the player is two 3rd in the radius, just apply the full knockback of the explosion
	if (dist < p->explosion_radius / 1.5) {
		dist_vector = vec3_multiply_float(dist_vector, p->knockback);
		player_push(dist_vector);
		player_damage(p->damage);
		return;
	}

	//if not, apply a graduate force on him, depending on the distance between him and the explosion
	
	//* 1.5 so it's a smooth transition between the two 3rd and the rest of the explosion
	float norm_dist = dist / (p->explosion_radius * 1.5);
	dist_vector = vec3_multiply_float(dist_vector, 1 - norm_dist);
	dist_vector = vec3_multiply_float(dist_vector, p->knockback);
	player_push(dist_vector);

	int norm_dmg = p->damage;
	norm_dmg *= vec3_length(dist_vector);
	player_damage(norm_dmg); //the results are pretty funky but it's good enough for now
}
